/////////////////////////////////////////////////////////////////////////////////
//
// Shader  : TheEmu - Complex Functions.fsh
// Creator : TheEmu
// Version : 1.10.8
// Date    : 2015/12/20
// License : Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
//
// History:
//   1.0    - 2015/06/05 - Initial release.
//   1.0.1  - 2015/06/07 - Split large switch statement into smaller ones
//                         because the NVIDIA fragment shader compiler is
//                         limited to 64 cases in a switch statement.
//   1.0.2  - 2015/06/20 - Added a few more functions.
//   1.0.3  - 2015/07/13 - Added a few more functions - total now 99.
//   1.1    - 2015/07/14 - Added ability to iterate functions.
//   1.2    - 2015/07/18 - Added FunctionMode and DataPanel.
//   1.3    - 2015/07/28 - Expanded DataPanel functionality.
//   1.3.1  - 2015/07/28 - Added not-a-number handling.
//   1.4    - 2015/08/03 - Provided versions with and without DataPanel support.
//   1.4.1  - 2015/08/10 - Expanded DataPanel functionality.
//   1.4.2  - 2015/08/13 - Minor optimisations.
//   1.5    - 2015/08/20 - Added DomainMap functionality.
//   1.6    - 2015/08/23 - Added DomainTrim functionality.
//   1.6.1  - 2015/08/24 - Moved test for NaN. Now properly tests fz itself.
//   1.7    - 2015/09/05 - Added DomainMapScaleMode, PreRotate and PostRotate.
//   1.7.1  - 2015/09/05 - Improved contour rendering for large R.
//   1.8    - 2015/09/12 - Added DomainMapMode codes 12 to 19.
//   1.8.1  - 2015/09/13 - Added ContourMode, ContourPowerScale and ContourFadeRate.
//   1.8.2  - 2015/09/14 - Reassigned ContourMode codes and changed its default.
//   1.8.3  - 2015/09/14 - Added the Translucency control parameter.
//   1.8.4  - 2015/09/19 - Corrected iteration count logic, now starts from 0.
//   1.9    - 2015/09/20 - Added TransitionMode and Transition fraction.
//   1.9.1  - 2015/09/21 - Minor tidy up.
//   1.9.2  - 2015/09/23 - Added contour modes 8 and 9.
//   1.9.3  - 2015/09/25 - Added ContourDamping and USE_IF_BASED_FUNCTION_SELECTION
//   1.9.4  - 2015/09/27 - Changed to a 6x5 font as this supports a full alphabet
//                         and added the SeriesLetter parameter (for the data panel)
//   1.10   - 2015/09/28 - Contour mode and density now used in all colour modes.
//   1.10.1 - 2015/10/01 - complex_functions_data_panel - changed ivec2 to vec2.
//   1.10.2 - 2015/10/09 - Added optional MAX_ITERATION_LIMIT
//   1.10.3 - 2015/10/10 - Fixed error in transitions of iterated functions.
//   1.10.4 - 2015/10/13 - Defaulting to not using USE_IF_BASED_FUNCTION_SELECTION
//                         as it slows down the NVIDIA GPU quite badly and at
//                         the moment it is not needed for the Intel GPU.
//   1.10.5 - 2015/10/16 - Data panel optimisations.
//   1.10.6 - 2015/10/18 - Added PrePower and PostPower control parameters.
//   1.10.7 - 2015/10/21 - Added USE_EMU_DATA_PANEL_PART_n options.
//   1.10.8 - 2015/12/10 - Reduced #define use - helps with Intel GPU.
//
/////////////////////////////////////////////////////////////////////////////////
//
// This shader is based on the Complex Domains shader by Huw Bowles but has been
// greatly modified and extended.  It  generates images based on considering the
// window as a rectangular region of the complex plane and evaluating a function
// at each point.  The function results are used to either display the structure
// of the function in a mathematical way or as a distortion applied to a texture
// which tiles the window. The results may be optionally animated.
//
// The operation of the shader is controled by a number of parameters  that  may
// specified by uniform: clauses in a VGHD scene file.  These parameters are all
// described in comments below.
//
// Note, only the variables declared as uniform are inputs,  the other variables
// declared with them are simply constants derived from them.
//
/////////////////////////////////////////////////////////////////////////////////
//
// This shader can be tailored by means of a set of optional #define  statements
// each of which modify the operation of the shader in ways that  are  described
// in detail in the following comments. The most general form of the shader will
// support all of its functionality and allow it to be controled using a set  of
// uniforms that may be optionaly defined in the scene file. However, this comes
// at the price of always including code that may not be needed for a particular
// use of the shader,  and  it is better to eliminate this code as it slows down
// starting the scene and increases the load on the GPU when it is running.  The
// slowing down of scene start up has been particularly noticed  when  using  an
// Intel Internal Graphics processor,  the effect being much less noticable when
// some more powerful GPU is used.  It  is thought that this is due to the Intel
// GLSL compiler performing extensive optimisations to allow the  less  powerful
// graphics processor to almost match the performance of much more powerfull GPU
// hardware.
//
// I have provided three standard tailorings of this shader, these being
//
//    TheEmu - Complex Functions.fsh
//    TheEmu - Complex Functions - No Data Panel.fsh
//    TheEmu - Complex Functions - Data Panel Only.fsh
//
// the first being the general form of the shader, the second eliminates support
// for the data panel and the third only generates a data panel.  These versions
// differ only in whether EVALUATE_FUNCTION and USE_EMU_DATA_PANEL are defined.
//
// The user may make use of these and other shader tailoring possibilities as is
// described below.
//
/////////////////////////////////////////////////////////////////////////////////

// EVALUATE_FUNCTION controls the primary functionality of this shader.  It may
// be left undefined, defined with no value or defined as function name.  If it
// is not defined then the shader's complex function evaluation code is omitted
// and only secondary functionality provided.  If defined without a value, i.e.
// using just
//
//    #define EVALUATE_FUNCTION
//
// then FunctionId, the first or only function to evaluate,  will be defined as
// a uniform int that can be assigned a value in the scene  file.  However,  if
// it is defined as a function name,  which  must  be of the form FUNC_nn where
// nn is one of the supported function ids,  then  only  that functions will be
// evaluated and no code will be generated for the other functions.  Because of
// problems with the GLSL preprocessor if EVALUATE_FUNCTION is defined in  this
// way then it is also necessary to define EVALUATE_FUNCTION_ID, e.g.
//
//   #define EVALUATE_FUNCTION FUNC_66
//   #define EVALUATE_FUNCTION_ID 66

#define EVALUATE_FUNCTION // Function Id to be specified via scene file.

// Optionaly an upper bound can be specified for the number of times  functions
// may be iterrated. Define this to reduce load on the GPU if some scenes cause
// the animation to judder.

// #define MAX_ITERATION_LIMIT 16 // Use a lower value to reduce GPU load.

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// If USE_EMU_DATA_PANEL is defined then data panel support will be provided by
// the shader. If it is defined without a value or with a value of 0,  i.e.  by
// using on of
//
//    #define USE_EMU_DATA_PANEL
//    #define USE_EMU_DATA_PANEL 0
//
// then the generation of the data panel is controled via the DataPanel uniform
// that can be assigned a value in the scene file. However, if a non-zero value
// is given when defining USE_EMU_DATA_PANEL, i.e. using
//
//    #define USE_EMU_DATA_PANEL 1
//
// then data panel generation is unconditional.

#define USE_EMU_DATA_PANEL // Data panel selectable from scene file.

// Using the data panel may load the system to the point where the  either  the
// animation of the results of the shader or of the VGHD clip it accompanies is
// slightly jerky. This can be reduced eliminating the data panel  entirely  by
// commenting out the definition of USE_EMU_DATA_PANEL or by reducing how  much
// data is displayed by commenting out one or more of the following lines.  The
// "most significant" part of the data is in part 1 of the panel and the "least
// significant" data is in part 3.  Even if all three have been commented out a
// minimal data panel with the "series letter" and current function id will  be
// generated.

#define USE_EMU_DATA_PANEL_PART_1
#define USE_EMU_DATA_PANEL_PART_2
#define USE_EMU_DATA_PANEL_PART_3

// By default the data panel will be permanent but it may be made to fade in or
// out by defining a panel fade cycle. If the data panel is being controled via
// the scene file then this fade cycle can also be specified there but  if  the
// selection of the data panel is itself permanent then the fade cycle, if any,
// has to be specified by defining USE_EMU_DATA_PANEL_CYCLE, for example by
//
//   #define USE_EMU_DATA_PANEL_CYCLE 20.0,3.0,10.0,1.0
//
// where the first vlue is the overall cycle time in seconds, the second is the
// delay before starting to fade in,  the  third is the subsequent delay before
// starting to fade out and the final value is the duration of fading.

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// The selection of which function to evaluate may be done either using 'switch'
// statements or by a series of 'if' statements.  The first of these is prefered
// as it is a much cleaner coding construct.  However,  this  shader is so large
// that it is at the limits of what can be handled by the complier or driver for
// the Intel Integrated Graphics Processor on my laptop and I have  to  simplify
// it a little in order to use it at all.  If I do not simplify it then the VGHD
// program crashes with the Intel GPU is used.  There  have  been no problems of
// this sort when using the same laptops's NVIDIA GPU.
//
// If the following symbol is defined then the shader will use the inferior 'if'
// based method for function slection, otherwise it uses switch statements.  You
// may uncomment it if your system has no problems with the switch based form.

// #define USE_IF_BASED_FUNCTION_SELECTION

/////////////////////////////////////////////////////////////////////////////////

// Standard shader inputs.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

  // VGHD's standard elapsed time and window size uniform inputs are not  used
  // directly in the body of the shader,  instead it uses the following copies
  // of them as this makes things easier if we want to modify them in some way
  // or use different sources for these parameters as would have to be done if
  // the shader was used elsewhere.
  //
  // A modified version of the elapsed time is used because u_Elapsed seems to
  // start from zero before the shaders are compiled. Using Intel's Integrated
  // Graphisc Processor compiling this shader takes a long time and  u_Elapsed
  // typically has a value between 15 and 25 seconds when the first pixels are
  // displayed.  For the NVIDIA GPU on the same laptop the delay is only about
  // 5 seconds. To compensate for this, and avoid losing too much of the first
  // part of the output, the time is adjusted.

  float Time = max ( u_Elapsed-15.0, 0.0 );
  vec2 WindowSize = u_WindowSize;

/////////////////////////////////////////////////////////////////////////////////

// Some useful constants and macros.

const vec2 vec2_0 = vec2 ( 0.0, 0.0 );
const vec2 vec2_1 = vec2 ( 1.0, 1.0 );

const vec4 vec4_0 = vec4 ( 0.0, 0.0, 0.0, 0.0 );
const vec4 vec4_1 = vec4 ( 1.0, 1.0, 1.0, 1.0 );

// DEFAULT is used for replacing any parameters that have not been explicitly
// specified, and hence are zero, with an appropriate default value.

#define DEFAULT(type,x,default_value) (x==type(0.0)) ? (default_value) : (x)

/////////////////////////////////////////////////////////////////////////////////

// Control inputs. Part 1 - Common control parameters.

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// FunctionId specifies which complex function is to be plotted.  See  the set of
// FUNC_nn definitions below for the list of functions and their ids.  FunctionId
// may be specified either by defining EVALUATE_FUNCTION_ID, which configures the
// shader to always evaluate a specific function,  or  it may be specified by the
// .scn file.

#ifdef EVALUATE_FUNCTION_ID
   // Fixed value FunctionId specifed by EVALUATE_FUNCTION_ID
   #define FunctionId EVALUATE_FUNCTION_ID
#else
   #if defined EVALUATE_FUNCTION || defined USE_EMU_DATA_PANEL
      // The FunctionId must be specified by the .scn file
      uniform int FunctionId;
   #else
      #error Invalid tailoring for FunctionId specification
   #endif
#endif

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// The Series letter, if specified, is currently only used to generate the first
// part of the data panel. Its value is interpreted as a short series of letters
// with 1,2,3..26 corresponding to A,B..Z, 101,102..126 as AA,AB..AZ etc.

uniform int SeriesLetter;

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// By default the function is simply func(z) with func specified by  FunctionId.
// However this can be modified by using a non zero value for FunctionMode which
// is interpreted as follows (the second column is the binary representation).
//
//    0 - 0000 - evaluate func(z)
//    1 - 0001 - evaluate func(log(z))
//    2 - 0010 - evaluate log(func(z))
//    3 - 0011 - evaluate log(func(log(z)))
//
//    4 - 0100 - evaluate func(1/z)
//    5 - 0101 - evaluate func(1/log(z))
//    6 - 0110 - evaluate log(func(1/z))
//    7 - 0111 - evaluate log(func(1/log(z)))
//
//    8 - 1000 - evaluate 1 / func(z)
//    9 - 1001 - evaluate 1 / func(1/z)
//   10 - 1010 - evaluate 1 / func(log(z))
//   11 - 1011 - evaluate 1 / func(1/log(z))
//
//   12 - 1100 - evaluate 1 / log(func(z))
//   13 - 1101 - evaluate 1 / log(func(1/z))
//   14 - 1110 - evaluate 1 / log(func(log(z)))
//   15 - 1111 - evaluate 1 / log(func(1/log(z)))
//
// Note, as log(1/z) = -log(z) we do not need to handle log of inverse as well
// as inverse of log as they are related only by a change of sign of x and y.

uniform int FunctionMode;

   bool logOfArgument  = ( FunctionMode & 1 ) != 0;
   bool logOfFunction  = ( FunctionMode & 2 ) != 0;
   bool invertArgument = ( FunctionMode & 4 ) != 0;
   bool invertFunction = ( FunctionMode & 8 ) != 0;

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// By default the functions are not iterated. However, this can be overridden by
// specifying non-zero values for one or both of MinIterate and MaxIterate which
// default to 1 and MinIterate respectively. When these parameters are used then
// each function will be iterated by each of the values in the specified range.

uniform int MinIterate;
uniform int MaxIterate;

   #ifndef MAX_ITERATION_LIMIT
      int minIterationCount = max ( MinIterate, 0 );
      int maxIterationCount = max ( MaxIterate, minIterationCount );
   #else
      int minIterationCount = min ( MAX_ITERATION_LIMIT, max ( MinIterate, 0 ) );
      int maxIterationCount = min ( MAX_ITERATION_LIMIT, max ( MaxIterate, minIterationCount ) );
   #endif

   int iterationRange = maxIterationCount - minIterationCount + 1;

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// By default only the function explicitly given by FunctionId is used. However,
// if a non-zero value for FunctionDuration is specified then  the  function  id
// will cycle through the available functions  changing  every  FunctionDuration
// seconds.  Note, if the function is iterated then this time will be split into
// equal portions for each iteration level.

uniform float FunctionDuration;

   float FunctionDurationX = DEFAULT(float,FunctionDuration,1.0e6);

   #ifdef EVALUATE_FUNCTION_ID
      #define staticFuncId true
   #else
      bool staticFuncId = FunctionDuration == 0.0;
   #endif

uniform float IterationDuration;

   float iterationTime = (IterationDuration > 0.0)
                           ? IterationDuration
                           : ( FunctionDurationX / float(iterationRange) );

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Optionally the function argument and result may both be modified by  applying
// constant scale factors and adding constants, i.e. the evaluation of fz = f(z)
// is replaced by
//
//          fz' = f ( ((z+a)*b)**c )
//          fz" = ((fz'+a')*b')**c'
//
// where a, b and c are constants modifying the domain of the function while a',
// b', c' modify its codomain, i.e. its result.
//
// Note, the +, * and ** operators above do NOT denote complex  operations  only
// but only scalar addition, multiplication and exponentiation of the individual
// x and y components of z.  Also note that the exponentiation operator x**c has
// been modified to mean sign(x)*pow(abs(x),c) rather than just pow(x,c) as this
// both preserves the sign and avoids numerical problems when x is negative.
//
// It may be noted that the effect of these four modifiers on the final image is
//
//    PreScale  - zooms in or out of and/or stretches the image.
//    PreAdd    - causes the center of the image to be shifted.
//    PrePower  - as PreScale but varies with position in image.
//    PostScale - alters scale of the contours and colour bands.
//    PostAdd   - affects the form of the function result.
//    PostPower - affects the form of the function result.
//
// The effect of using PrePower is more complex than the other modifiers. As has
// stated above it introduces a zoom factor that varies with the position in the
// function's domain (and hence in the final image) as a result it also distorts
// the image.  Its  default value of 1.0 has no effect,  values larger than this
// magnify the area round the origin and shrink areas more that a unit  distance
// from the origin, the details varying somewhat with the domain mapping mode. A
// useful side effect of this is that for those domain mappings that depend on R
// (see DomainMapMode below) the mapping to circles is distorted to members of a
// family of shapes depending on the value of  PrePower.  These  shapes  include
// rounded squares when PrePower is greater than 1.0 and rounded  diamonds  when
// PrePower is between 0.5 and 1.0. PrePower should normally be in the range 0.2
// to 3.0, though values outside of this range, other than 0.0, are accepted.
//
// In addition rotations may be applied to the domain and codomain.  A  rotation
// of the domain rotates the image   (or each part of the image that corresponds
// to a copy of the whole complex plane) while rotating the codomain  will  only
// affect the colouring.

uniform vec2  PreAdd;     // The constant a, defaults to (0.0,0.0)
uniform vec2  PreScale;   // The constant b, defaults to (1.0,1.0)
uniform float PreRotate;
uniform float PrePower;

uniform vec2  PostAdd;    // The constant c, defaults to (0.0,0.0)
uniform vec2  PostScale;  // The constant d, defaults to (1.0,1.0)
uniform float PostRotate;
uniform float PostPower;

   vec2 DomainScale   = DEFAULT ( vec2, PreScale,  vec2_1 );
   vec2 CodomainScale = DEFAULT ( vec2, PostScale, vec2_1 );

   vec2 DomainPower   = vec2 ( DEFAULT ( float, PrePower,  1.0 ) );
   vec2 CodomainPower = vec2 ( DEFAULT ( float, PostPower, 1.0 ) );

   float PreAlpha  = radians(PreRotate);
   float PostAlpha = radians(PostRotate);

   float SinPreAlpha = sin(PreAlpha);
   float CosPreAlpha = cos(PreAlpha);

   mat2 DomainRotation = mat2 ( CosPreAlpha, -SinPreAlpha,
                                SinPreAlpha,  CosPreAlpha
                              );

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// By default the mapping from the window coordinates to the complex numbers is,
// after PreAdd and PreScale (see above) and ignoring the overall scale  factor,
// just (x,y) => (x,y) such that (0,0) is the center  of  the window.  Alternate
// mappings may be selected using the DomainMapMode control parameter  according
// to the table below.  Note,  these are mapping from screen co-ordinates to the
// complex number that they represent, because that is what the shader does, and
// each point on the screen maps to a single unique complex number.  The reverse
// mapping, from complex number to screen position,  is not always so simple and
// in many cases a single complex number maps to more than one screen  position.
// See below for a little more detail.
//
//          Coord => Complex number
//
//      0 : (x,y) => K * (  x,   y  )
//      1 : (x,y) => K * ( 1/x, 1/y )
//
//      2 : (x,y) => K * (  x*R,     y*R    )
//      3 : (x,y) => K * ( 1/(x*R), 1/(y*R) )
//
//      4 : (x,y) => K * ( x*F(x), y*F(y) )
//      5 : (x,y) => K * ( x/F(x), y/F(y) )
//
//      6 : (x,y) => K * ( x*F(R), y*F(R) )
//      7 : (x,y) => K * ( x/F(R), y/F(R) )
//
//      8 : (x,y) => K * ( x*tan(x'), y*tan(y') )
//      9 : (x,y) => K * ( x/tan(x'), y/tan(y') )
//
//     10 : (x,y) => K * ( x*tan(R'), y*tan(R') )
//     11 : (x,y) => K * ( x/tan(R'), y/tan(R') )
//
//     12 : (x,y) => K * ( x*abs(F(x)), y*abs(F(y)) )
//     13 : (x,y) => K * ( x/abs(F(x)), y/abs(F(y)) )
//
//     14 : (x,y) => K * ( x*abs(F(R)), y*abs(F(R)) )
//     15 : (x,y) => K * ( x/abs(F(R)), y/abs(F(R)) )
//
//     16 : (x,y) => K * ( x*abs(tan(x')), y*abs(tan(y')) )
//     17 : (x,y) => K * ( x/abs(tan(x')), y/abs(tan(y')) )
//
//     18 : (x,y) => K * ( x*abs(tan(R')), y*abs(tan(R')) )
//     19 : (x,y) => K * ( x/abs(tan(R')), y/abs(tan(R')) )
//
// where v' is abs(v)/(pi/2), F(x) is abs(x)/(1-abs(x)), R is length(x,y) and K
// is a constant scale factor.  The factor of pi/2 is used because tan(pi/2) is
// infinite and we want infinity to map to either 0.0 or 1.0.
//
// Note, if 1 and 2 are rewritten as
//
//      (x,y) => K * ( x/(x*x),   y/(y*y)   )
//      (x,y) => K * ( x/(x*x*R), y/(y*y*R) )
//
// then all of the above mappings can be seen to have the same general form
//
//      (x,y) => K * ( x * G(x), y * G(y) )
//
// where G(x) is some function of x that may also involve R.
//
// The first four mappings are simple one to one mappings of the  whole  complex
// plane to and from an infinite plane,  i.e, each complex number is mapped to a
// a single point most of which will be beyond the screen's boundaries. For maps
// 0 and 2 the visible portion corresponds to complex numbers close to (0.0,0.0)
// and for maps 1 and 3 the visible portion corresponds to complex numbers  that
// are "near" infinity with the points at infinity mapping to (0,0).
//
// The remaining mappings are not so simple and  each  maps  the  whole  of  the
// complex plane to a finite region centered at (0,0). In the case of maps 4 and
// 5 this is a square of side 2 and for maps 6 and 7 it is a circle of radius 1.
// For maps 4 and 6 the edge of this square or circle represents the  points  at
// infinity in the complex plane while for maps 5 and 7 the edges of the  square
// or circle represent (0.0,0.0) and the center represents infinity.
//
// Maps 8 to 11 are similar to maps 4 to 7 but use a different function,  tan(x)
// rather than 1/x, to map between finite and infinite domains.  For the central
// square or circle, i.e. the primary mapping of the infinite complex plane, the
// difference is minor but for those parts of the screen  outside  this  primary
// area the differences are profound. For 4 to 7 those parts outside the central
// square or circle will be a single copy of the central primary region with the
// copy stretching out to infinity in all directions while for maps  8 to 11 the
// outer regions will comprise a series of square or circular bands each of  the
// same width, 2, surrounding the central region with each outer band holding  a
// stretched copy of the central region.
//
// Maps 12 to 19 are variations of maps 4 to 11 differing from them in  the  way
// in which the copies of the inner region are made.
//
// By default the mapping scale factor,  K in the above expressions,  is  chosen
// such that the x axis includes the range -pi/1.3 to pi/1.3 with the same scale
// factor used for both x and y.  This  ensures  that (-pi/2,0) and (pi/2,0) are
// included in the window as these are often interresting  points.  However  the
// scaling can be modified in two ways, these being by specifying the scale mode
// via DomainMapScaleMode or additional scale factors via DomainMapScale.  Valid
// values for DomainMapScaleMode and their interpretations are
//
//     0 - Default scaling as described above
//
//     1 - Scale such that (-1,-1) and (1,1) are at the  window's  edge
//         while keeping equal x and y scales.
//
//     2 - Scale such that (-pi,-pi)/1.3 and (pi,pi)/1.3 are the bottom
//         left and top right corners of the window.
//
//     3 - Scale such that (-1.0,-1.0) and (1.0,1.0)/ are at the bottom
//         left and top right corners of the window.
//
//     4 - Scale such that any trimmed region, see below, just fits the
//         window while keeping equal x and y scales. Only valid if the
//         domain trimming functionality is used.
//
//     5 - Scale such that any trimmed region, see below, just fits the
//         window without keeping equal x and y scales. Only valid when
//         the domain trimming functionality is used.
//
// In all cases an addtional scale factor may also be given using DomainMapScale.
// If it is not specified, or is specified as 0.0, then a value of 1.0 is used.

uniform int   DomainMapMode;
uniform vec2  DomainMapScale;
uniform float DomainMapRotation;
uniform int   DomainMapScaleMode;

    vec2 DomainMapScaling
      = vec2 ( DEFAULT ( float, DomainMapScale.x, 1.0),
               DEFAULT ( float, DomainMapScale.y, 1.0)
             );

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// After applying the mapping specified by DomainMapMode the function's domain
// may be trimmed by specifying using the DomainTrimMode and DomainTrimParams.
//
// Valid values for DomainTrimMode are
//
//     0 - No trimming
//     1 - Retain a square or rectangular area.
//     2 - Discard a square or rectangular area.
//     3 - Retain a diamond shaped area.
//     4 - Discard a diamond shaped area.
//     5 - Retain a circular or eliptical area.
//     6 - Discard a circular or eliptical area.
//
// For all of these trim modes the first two components of DomainTrimParams are
// the (x,y) components of the center of the area to be trimmed and the  second
// two components give the size in the x and y directions.

uniform int DomainTrimMode;
uniform vec4 DomainTrimParams;

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// By default an internal colouring scheme is used, however this may be replaced
// by an input texture by specifying a non zero value for ColourScheme which may
// take one of the following values
//
//    0 - Internal colourisation based on cartesian components (x,y)
//    1 - Internal colourisation based on polar components (r,theta)
//    2 - Internal colourisation based on polar components (r,theta)
//
//   10 - Texture based colourisation based on cartesian components (x,y)
//   11 - As 10 but with no tile flipping (see below)
//   12 - Texture based colourisation based on polar components (r,theta)
//   13 - As 12 but with no tile flipping (see below)
//
// where (x,y) are the real and imaginary parts of the complex  number z = x+i*y
// and (r,a) are the polar components of the same compex number z = r*exp(i*a).
//
// When using the texture based colouring schemes it is normal to flip alternate
// tiles so that there is at least continuity of colour where the tiles meet  as
// this produces a smoother image.  However, in some cases, this may be unwanted
// and so versions of these colouring schemes that do not perform this  flipping
// are also provided and these are useful when  the  textures  are  deliberately
// directional, e.g. have an arrow on them.

uniform int ColourScheme;

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// By default the output is static, however this may be changed to a periodically
// varying colourisation by specifying its period in seconds.

uniform float ColourPeriod;

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// By default the transition between the images for one function and the next is
// instantaneous. However this may be changed to gradualy transition between the
// previous image and that for the current function by specifying  a  transition
// mode and transition time.
//
// The transition mode is specified by TransitionMode which may take any of  the
// following values
//
//      0 - Default, instantaneous transitions.
//
//      1 - Fade out the old image while fading in the new.
//
//      2 - Interpolate intermediate function values.
//
// Except for TransitionMode 0 the duration of the transitions is controlled  by
// the TransitionFraction parameter. This specifies the fraction of the function
// period or the function iteration period is is used for the  transition.  Only
// values in the range 0.0 to 1.0 should be used with 0.0  corresponding  to  an
// instantaneous transition and 1.0 corresponds to the whole of the function  or
// iteration period being used for the transition.

uniform int   TransitionMode;
uniform float TransitionFraction;

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// The shader does not display the values of any  parameters,  but  if  DataPanel
// is set to a non zero value then a small panel in the bottom right corner  will
// be used to display the values of a set of important control parameters.  Refer
// to the body of the shader for details.
//
// By default the data panel will neither fade in or out,  but  if DataPanelCycle
// is defined then it will so in according with the DataPanelCycle parameters.
//
// If the value is positive then the data panel will ovelay the main image, it it
// is negative then only the data panel is generated
//
// Note, currently the absolute value of DataPanel has no significance but future
// versions of this shader may use it to control the position or contents of  the
// panel so only use 0, 1 or -1.

#ifdef USE_EMU_DATA_PANEL

   #if USE_EMU_DATA_PANEL+0 == 0
      uniform int DataPanel;
   #else
      #define DataPanel USE_EMU_DATA_PANEL
   #endif

   #ifdef USE_EMU_DATA_PANEL_CYCLE
      #define DataPanelCycle vec4(USE_EMU_DATA_PANEL_CYCLE);
   #else
      uniform vec4 DataPanelCycle;
      #define USE_EMU_DATA_PANEL_CYCLE
   #endif

#endif

/////////////////////////////////////////////////////////////////////////////////

// Control inputs. Part 2 - Parameters for the internal colour schemes.

// The internal colour scheme chooses a colour based on the theta angle  of  the
// polar representation, (R,theta) of the function value.  However, this is then
// modified by optionaly plotting contours of the cartesian, (x,y) or the  polar
// representations. By default contours are only plotted for x, y and R but this
// can be altered using the ContourDensity control parameter, see below.

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Contours may be drawn using linear, logarithmic or arc-tangent scales.  These
// are selected via the ContourMode control parameter which is a 2 digit number,
// AB, where a controls the x and y contours while b controls the R contours.  A
// and b are interpreted according to the following table.
//
//     0 : default
//     1 : linear
//     2 : log
//     3 : atan
//     8 : 0+inf
//     9 : none
//
// where 'none' indicates that the corresponding contour will be suppressed  and
// 0+inf represents a variation on the atan (arc-tangent) scaling mode in which,
// with the default values for ContourPowerScale  and  ContourScale,  will  only
// produce contours at 0 with a gradual fading to dark, for x and y, or to white
// for R as the value approaches infinity.
//
// The default contour mode is also a variation on the arc-tangent scaling using
// x => 4.0*pow(atan(x),2.0) for positive x and its negative for negative x.  It
// combines simple arc-tangent scaling with ContourPowerScale  and  ContourScale
// both having their values doubled - see below for these scale modifiers.
//
// A linear scale is always used for the theta contours if they are enabled.
//
// The use of linear scaling for R often results in the contours being "crowded"
// when R is large as it will then also normaly be rapidly changing.  By using a
// logarithmic scale this effect is greatly reduced and the images are improved.
// For this reason it has been chosen as the default for the R contours.
//
// There is less of a reason to use logarithmic scaling for x and y,  but it has
// also been chosen as the default for these contours - mainly because the final
// images simply look more interresting, at least to me.
//
// As an alternative to the use of logarithmic scales arc-tangent scaling may be
// used instead.  This  has the advantage of eliminating all contour crowding at
// high values of x, y or R but at the expense of contours being very wide apart
// these high values.  With a simple arc-tangent scaling and the default contour
// of density of 1.0,  see below,  contours will only generated for -1, 0 and +1
// which  represent infinity, zero and +infinity respectively. More contours are
// generated if the contour density is increased with 4 being a reasonable value
// to use.  For  this  reason if arc-tangent scaling is selected by default then
// the effective contour density is increased by a factor of four,  however this
// is not done if arc-tangent scaling is explicitly selected.
//
// Although contours are not added when textured colouring is used  the  contour
// mode and contour density are still used to control the tile sizes used by the
// textured colouring algorithm, in effect the (x,y) contours become  the  edges
// of the tiles.

uniform int ContourMode;

   int xyContourMode = ContourMode / 10;
   int rContourMode  = ContourMode - xyContourMode*10;

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// After applying the scalings implied by ContourMode a further scaling  may  be
// applied by raising each scaled function value to some power specified via the
// ContourPowerScale control parameter. The default, which is also used if it is
// explicitly specified as (0.0,0.0,0.0,0.0), is (1.0,1.0,1.0.1.0) which has  no
// effect.  Reasonable values for the components of ContourPowerScale are in the
// range 0.25 to 2.0.  Values outside this range may be used, including negative
// values, a value close to zero should be avoided as pow(x,0.0) is always 1.0.

uniform vec4 ContourPowerScale;

   vec4 ContourPowerFactor
     = vec4 ( DEFAULT ( float, ContourPowerScale.x, 1.0),
              DEFAULT ( float, ContourPowerScale.y, 1.0),
              DEFAULT ( float, ContourPowerScale.z, 1.0),
              DEFAULT ( float, ContourPowerScale.w, 1.0)
            );

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// By default the contour each lines represent steps of 1.0 in the components of
// the function's value after applying the  ContourMode  and  ContourPowerFactor
// scaling factors.  This spacing can be changed by specifying a non-zero values
// for ContourDensity,  the  four components of which give the number of contour
// lines per unit step in the corresponding components of the scaled value.
//
// In the order in which they must be given in the  uniform:  clause in the VGHD
// scene file the components of ContourDensity control the x,y,r,theta spacings.
//
// As a special case if all four components of ContourDensity are 0.0 is treated
// as if it were (1.0,1.0,1.0,0.0) so that the default behaviour is to plot x, y
// and r contours but not those for theta. This is mainly for aesthetic reasons.
//
// If all four sets of contours are to be suppressed then use a  negative  value
// for at least one of ContourDensity's components.
//
// Normally the components of ContourDensity should be small integers (up to 10)
// as larger values usualy result in the contour lines being too crowded. It is
// valid to use non-integer values, but that complicates their interpretation.
//
// Note, if the theta contours are enabled by specifying a density of 4 for them
// then these will lie along the x and y axes and be seen instead of the x=0 and
// y=0 contours - in effect highlighting these particular x and y contours.
//
// The default dark and white colouring used for the contours may be ameliorated
// by specifying a non-zero value for ContourDamping. This value for this should
// be in the range 0.0, for no damping, to 1.0 for full damping.  A value of 0.1
// will reduce the intensity of the the contours which can be a bit "glaring" if
// the default undammped intensity is used.

uniform vec4 ContourDensity;
uniform vec4 ContourDamping;

   vec4 ContourXX = DEFAULT ( vec4, ContourDensity, vec4(1.0,1.0,1.0,0.0) );

   vec4 ContourIntensity = vec4 ( (ContourXX.x>0.0) ? 2.0 : 0.0, // Using 2.0 here avoids
                                  (ContourXX.y>0.0) ? 2.0 : 0.0, // needing a factor of 2
                                  (ContourXX.z>0.0) ? 2.0 : 0.0, // when ContourIntensity is
                                  (ContourXX.w>0.0) ? 2.0 : 0.0  // used in the code.
                                ) * ( 1.0 - ContourDamping );

   vec4 ContourScale = vec4 ( (ContourXX.x>0.0) ? ContourXX.x : 1.0,
                              (ContourXX.y>0.0) ? ContourXX.y : 1.0,
                              (ContourXX.z>0.0) ? ContourXX.z : 1.0,
                              (ContourXX.w>0.0) ? ContourXX.w : 1.0
                             );

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// By default the contour lines are somewhat blurred for  artistic  effect.  The
// blurring can be changed by specifying a non zero value for  ContourSharpness.
// The first component of this parameter controls the sharpness of  the  contour
// drawn for the cartesian components of the function's  value  and  the  second
// controls the sharpness of the contour for the polar components. Higher values
// increase the sharpness with values of 10.0 produceing very fine lines and low
// values cause the contour lines to broaden and to increase in  fuzzyness.  The
// default value is 1.0 which is a compromise between artistry and precision.

uniform vec2 ContourSharpness;

   vec4 ContourSharp = ( vec4 ( DEFAULT(vec2,ContourSharpness.xx,vec2_1),
                                DEFAULT(vec2,ContourSharpness.yy,vec2_1)
                              )
                       ) * 5.0;

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// When contours are too closely spaced in the image then the image quality will
// suffer from three effects, these being
//
//    1 - contours merging into larger areas
//    2 - the emergence of moire patterns
//    3 - intermitent or stripey contours
//    4 - "noisy" and "shimmering" areas
//
// The first effect can be reduced by using thinner contours, but that increases
// the imapact of other three.  A higher resolution would allow thinner contours
// to be used as would anti-aliasing techniques based on over sampling, but both
// of these solutions are expensive. Instead, as a cheap means of mitigating the
// effects, the contours are made to gradualy fade as R increases.  The  default
// fading seems to effective in most cases,  but can be overridden by specifying
// non-zero values for ContourFadeRate.  The  first component of ContourFadeRate
// modifies the fading for the (x,y) contours and the second modifies the fading
// for the (R,theta) contours.  Negative values for ContourFadeRate are replaced
// by 0.0.
//
// The default for ContourFadeRate, which is also used if it is explicitly given
// as (0.0,0.0), depends on the ContourMode.  It  causes  moderate fading except
// for contours for using arc-tangent scaling. For arc-tangent scaling where the
// default is to use no fading.
//
// Because (0.0,0.0) is replaced by the default ContourFadeRate it is  necessary
// to use (-1.0,-1.0) to explicitly suppress contour fading.
//
// Normal values for the components of ContourFadeRate are in the range 1 to 10.

uniform vec2 ContourFadeRate;

   vec2 DefaultFadeRate = vec2 ( ((xyContourMode==0)||(xyContourMode==3)) ? 0.0 : 1.0,
                                  ((rContourMode==0)||(rContourMode==3))  ? 0.0 : 1.0
                               );

   vec4 Contour_FadeRate = vec4 ( ( DEFAULT(vec2,ContourFadeRate.xx,DefaultFadeRate.xx) ),
                                  ( DEFAULT(vec2,ContourFadeRate.yy,DefaultFadeRate.yy) )
                                ) * -0.00003;

 // -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// By default the full spectrum red, yellow, green, cyan, blue, magenta, red  is
// mapped to values of theta corresponding to the range 0 to 360  degrees, which
// is regarded as a unit turn. This can be changed by specifying a no zero value
// for the ColourScale parameter.  Normaly  this should be a small integer value
// in which case it gives the number of times the spectrum will be repeated  for
// a full 360 degree change in theta. Its default value is 1.0.

uniform float ColourScale;

   float ColourScaleFactor = DEFAULT ( float, ColourScale, 1.0 );

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// By default any areas of the image that have been trimmed will be rendered  as
// transparent. This is changed by specifing a different colour for TrimColour.

uniform vec4 TrimColour; // Defaults to vec4_0

// By default any function value evaluated as NaN (not a number)  will  will  be
// rendered as transparent. This can be changed by specifying a different colour
// using NanColour. NaNs are generated by invalid floating point operations such
// such as can arise when values go out of range.

uniform vec4 NanColour; // Defaults to vec4_0

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// By default the generated image will be opaque except for any areas that  have
// been rendered using TrimColour or NanColour. his can be changed by specifying
// a non-zero value between 0.0 and 1.0 for Translucency with 0.0  corresponding
// to fully opaque and 1.0 to fully transparent.  This  parameter is not applied
// to any areas that have been rendered using TrimColour or NanColour, but their
// opacities may be directly specified their fouth component.

uniform float Translucency;

/////////////////////////////////////////////////////////////////////////////////

#ifdef EVALUATE_FUNCTION

// Control inputs. Part 3 - Parameters for the texture based colour schemes.

// The ColouringTexture input is only used if one of the texture based  colouring
// schemes is selected.  If required it is specified indirectly in the scene file
// as a source image for whatever node the shader is being used from,  i.e.  by a
// source: clause.

uniform sampler2D ColouringTexture; // Specified using a source: clause.

#endif

/////////////////////////////////////////////////////////////////////////////////

// Created by David Bargo - davidbargo/2015. Modified by TheEmu.
// License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

// -----------------------------------------------------------------------------

#ifdef EVALUATE_FUNCTION

// A few useful constants.

float pi_by_2 = radians( 90.0);
float pi      = radians(180.0);
float two_pi  = radians(360.0);

const vec2 z_i   = vec2(0.0,1.0);
const vec2 z_ONE = vec2(1.0,0.0);
const vec2 z_TWO = vec2(2.0,0.0);
const vec2 z_SIX = vec2(6.0,0.0);

// -----------------------------------------------------------------------------

// If some problem occurs when calculating f(z) then one of its components  may
// have a special value NaN, meaning "not a number".  This  indicates something
// like 0.0 / 0.0 occurred or that a mathematical function's argument  was  out
// of its valid range. GLSL provides an isnan function to test for NaNs but  it
// is more convenient to use the following macro instead. It uses the fact that
// a NaN is not equal to anything, even itself. It will evaluate to true if any
// component of x is a NaN.

#define IS_NAN(x) ( x != x )

// -----------------------------------------------------------------------------

// Some basic functions. These are used in the set of functions to be plotted.

vec2 iz ( vec2 c ) { return vec2 ( -c.y, c.x ); }

vec2 mulz ( vec2 c, vec2 d ) { return c * mat2(d.x,-d.y,d.y,d.x); }
vec2 mulz ( vec2 c, vec2 d, vec2 e ) { return mulz ( c, mulz(d,e) ); }

vec2 divz ( vec2 c, vec2 d ) { return c * mat2(d.x,d.y,-d.y,d.x) / dot(d,d); }

vec2 invz ( vec2 c ) { return vec2 ( c.x,-c.y ) / dot(c,c); }

vec2 sinz ( vec2 c ) { return vec2 ( sin(c.x)*cosh(c.y),  cos(c.x)*sinh(c.y) ); }
vec2 cosz ( vec2 c ) { return vec2 ( cos(c.x)*cosh(c.y), -sin(c.x)*sinh(c.y) ); }
vec2 tanz ( vec2 c ) { return divz ( sinz(c), cosz(c) ); }
vec2 secz ( vec2 c ) { return invz ( cosz(c) ); }
vec2 cscz ( vec2 c ) { return invz ( sinz(c) ); }
vec2 cotz ( vec2 c ) { return invz ( tanz(c) ); }

vec2 sinhz ( vec2 c ) { return sinz ( -iz(c) ); } // sinh(z) = sin(i*z)/i
vec2 coshz ( vec2 c ) { return cosz ( +iz(c) ); } // cosh(z) = cos(i*z)
vec2 tanhz ( vec2 c ) { return divz ( sinhz(c), coshz(c) ); }
vec2 sechz ( vec2 c ) { return invz ( coshz(c) ); }
vec2 cschz ( vec2 c ) { return invz ( sinhz(c) ); }
vec2 cothz ( vec2 c ) { return invz ( tanhz(c) ); }

vec2 sqrtz ( vec2 c ) { float n = length(c)+c.x; return vec2(n,c.y)/sqrt(2.0*n); }

vec2 logz  ( vec2 c ) { return vec2 ( log(sqrt(dot(c,c))), atan(c.y,c.x) ); }
vec2 exp2z ( vec2 c ) { return vec2 ( c.x*c.x - c.y*c.y, 2.0*c.x*c.y ); }
vec2 epowz ( vec2 c ) { return vec2 ( cos(c.y), sin(c.y))*exp(c.x); }

vec2 sum_of_negative_powers ( vec2 c, int n )
 { vec2 t1 = invz(c);
   vec2 t2 = t1;
   vec2 result = t1;
   for ( int i=0; i<n; i++ )
    { t2 = mulz(t2,t1);
      result += t2;
    }
   return result;
 }

// -----------------------------------------------------------------------------

#endif // EVALUATE_FUNCTION

// If you add more functions remember to update the LAST_FUNC_ID and to add
// the function evaluation to the list in the switch statement in func.

#define LAST_FUNC_ID 99 // REMEMBER TO UPDATE THIS

float functionT1 = Time / FunctionDurationX;

ivec2 functionIdX0 = staticFuncId
                      ? ivec2(0)
                      : ivec2(max(functionT1-1.0,0.0),functionT1);

ivec2 functionIdX = ivec2 ( mod ( functionIdX0+FunctionId-1, float(LAST_FUNC_ID) ) ) + 1;

float functionT2  = staticFuncId
                     ? Time
                     : Time - (functionIdX.x-1)*FunctionDurationX;

float iterationNf = mod ( functionT2 / iterationTime, iterationRange );

int currentIteration = minIterationCount + int(iterationNf);

float ColourPhase = PostAlpha
                     + ( (ColourPeriod==0.0) ? 0.0
                         : (Time*radians(360.0)/ColourPeriod)
                       );

float SinColourPhase = sin(ColourPhase);
float CosColourPhase = cos(ColourPhase);

mat2 ColourPhaseMatrix = mat2 ( CosColourPhase, -SinColourPhase,
                                SinColourPhase,  CosColourPhase
                              );

bool Transitioning = TransitionMode != 0
                  && fract(iterationNf) < TransitionFraction;

float TransitionFactor = smoothstep(0.0,TransitionFraction,fract(iterationNf));

// -----------------------------------------------------------------------------

#ifdef EVALUATE_FUNCTION

// Evaluate the currently selected function. If you add any new functions then
// you will need to add them here as well as declaring it above.

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

  // When using my laptop running VGHD with its Intel Integrated Graphics
  // processor then there is no problem with a single switch statement to
  // handle all the functions, currently 99 of them.  But, when using the
  // Nvidia GPU there is a limitation and only up to 64 functions can  be
  // handled by a single switch statement.  The  error reported being "IF
  // statement nested too deeply". To overcome this I use multiple switch
  // statements,  and  in case some other GPU driver has smaller limits I
  // handle the functions in groups of 16 and handle each such group in a
  // separate function. Groups of 16 are used rather than 10 or 20 in the
  // hope that the compiler is able to generate better code for a  switch
  // if the cases run from 0 to 2**n-1 as that is the  best  case  for  a
  // binary search which is how Nvidia's compiler handles it.

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

#ifndef staticFuncId

void func00 ( int id, vec2 z, inout vec2 fz )
 {

   #ifndef USE_IF_BASED_FUNCTION_SELECTION

     switch ( id )
      { case  0: fz = vec2_0;                                    break;
        case  1: fz = z;                                         break;
        case  2: fz = z * z;                                     break;
        case  3: fz = invz ( z );                                break;
        case  4: fz = invz  ( z + z_ONE ) + invz  ( z - z_ONE ); break;
        case  5: fz = invz  ( z + z_ONE ) - invz  ( z - z_ONE ); break;
        case  6: fz = invz  ( sqrtz(z) );                        break;
        case  7: fz = sqrtz ( z );                               break;
        case  8: fz = sqrtz ( z_ONE + z );                       break;
        case  9: fz = sqrtz ( z_ONE + z ) + sqrtz ( z_ONE - z ); break;
        case 10: fz = sqrtz ( z_ONE + z ) - sqrtz ( z_ONE - z ); break;
        case 11: fz = epowz ( z );                               break;
        case 12: fz = epowz ( invz(z) );                         break;
        case 13: fz = exp2z ( z );                               break;
        case 14: fz = exp2z ( invz(z) );                         break;
        case 15: fz = invz  ( z_ONE + epowz(z.yx) );             break;
      }

   #else

      if ( id ==  0 ) fz = vec2_0;
      if ( id ==  1 ) fz = z;
      if ( id ==  2 ) fz = z * z;
      if ( id ==  3 ) fz = invz ( z );
      if ( id ==  4 ) fz = invz  ( z + z_ONE ) + invz  ( z - z_ONE );
      if ( id ==  5 ) fz = invz  ( z + z_ONE ) - invz  ( z - z_ONE );
      if ( id ==  6 ) fz = invz  ( sqrtz(z) );
      if ( id ==  7 ) fz = sqrtz ( z );
      if ( id ==  8 ) fz = sqrtz ( z_ONE + z );
      if ( id ==  9 ) fz = sqrtz ( z_ONE + z ) + sqrtz ( z_ONE - z );
      if ( id == 10 ) fz = sqrtz ( z_ONE + z ) - sqrtz ( z_ONE - z );
      if ( id == 11 ) fz = epowz ( z );
      if ( id == 12 ) fz = epowz ( invz(z) );
      if ( id == 13 ) fz = exp2z ( z );
      if ( id == 14 ) fz = exp2z ( invz(z) );
      if ( id == 15 ) fz = invz  ( z_ONE + epowz(z.yx) );

   #endif

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

void func16 ( int id, vec2 z, inout vec2 fz )
 {

   #ifndef USE_IF_BASED_FUNCTION_SELECTION

      switch ( id - 16 )
       {
         case  0 : fz = invz ( z_ONE - epowz(z.yx) ); break;
         case  1 : fz = logz ( z );                   break;
         case  2 : fz = logz ( invz(z) );             break;
         case  3 : fz = logz ( z + invz(z) );         break;
         case  4 : fz = logz ( z - invz(z) );         break;
         case  5 : fz = logz ( z_ONE + invz(z) );     break;
         case  6 : fz = logz ( z_ONE - invz(z) );     break;
         case  7 : fz = sinz ( z );                   break;
         case  8 : fz = cosz ( z );                   break;
         case  9 : fz = tanz ( z );                   break;
         case 10 : fz = secz ( z );                   break;
         case 11 : fz = cscz ( z );                   break;
         case 12 : fz = cotz ( z );                   break;
         case 13 : fz = sinz ( sinz(z) );             break;
         case 14 : fz = sinz ( cosz(z) );             break;
         case 15 : fz = sinz ( tanz(z) );             break;
       }

   #else

      if ( id == 16 : fz = invz ( z_ONE - epowz(c.yx) );
      if ( id == 17 : fz = logz ( z );
      if ( id == 18 : fz = logz ( invz(z) );
      if ( id == 19 : fz = logz ( z + invz(z) );
      if ( id == 20 : fz = logz ( z - invz(z) );
      if ( id == 21 : fz = logz ( z_ONE + invz(z) );
      if ( id == 22 : fz = logz ( z_ONE - invz(z) );
      if ( id == 23 : fz = sinz ( z );
      if ( id == 24 : fz = cosz ( z );
      if ( id == 25 : fz = tanz ( z );
      if ( id == 26 : fz = secz ( z );
      if ( id == 27 : fz = cscz ( z );
      if ( id == 28 : fz = cotz ( z );
      if ( id == 29 : fz = sinz ( sinz(z) );
      if ( id == 30 : fz = sinz ( cosz(z) );
      if ( id == 31 : fz = sinz ( tanz(z) );

   #endif

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

void func32 ( int id, vec2 z, inout vec2 fz )
 {
   #ifndef USE_IF_BASED_FUNCTION_SELECTION

      switch ( id - 32 )
       {
         case  0 : fz = sinz ( secz(z) );          break;
         case  1 : fz = sinz ( cscz(z) );          break;
         case  2 : fz = sinz ( cotz(z) );          break;
         case  3 : fz = mulz ( sinz(z), sinz(z) ); break;
         case  4 : fz = mulz ( sinz(z), cosz(z) ); break;
         case  5 : fz = mulz ( sinz(z), tanz(z) ); break;
         case  6 : fz = mulz ( sinz(z), cotz(z) ); break;
         case  7 : fz = mulz ( sinz(z), secz(z) ); break;
         case  8 : fz = tanz ( sinz(z) );          break;
         case  9 : fz = tanz ( cosz(z) );          break;
         case 10 : fz = tanz ( tanz(z) );          break;
         case 11 : fz = tanz ( secz(z) );          break;
         case 12 : fz = tanz ( cscz(z) );          break;
         case 13 : fz = tanz ( cotz(z) );          break;
         case 14 : fz = sinhz ( z );               break;
         case 15 : fz = coshz ( z );               break;
       }

   #else

      if ( id == 32 ) fz = sinz ( secz(z) );
      if ( id == 33 ) fz = sinz ( cscz(z) );
      if ( id == 34 ) fz = sinz ( cotz(z) );
      if ( id == 35 ) fz = mulz ( sinz(z), sinz(z) );
      if ( id == 36 ) fz = mulz ( sinz(z), cosz(z) );
      if ( id == 37 ) fz = mulz ( sinz(z), tanz(z) );
      if ( id == 38 ) fz = mulz ( sinz(z), cotz(z) );
      if ( id == 39 ) fz = mulz ( sinz(z), secz(z) );
      if ( id == 40 ) fz = tanz ( sinz(z) );
      if ( id == 41 ) fz = tanz ( cosz(z) );
      if ( id == 42 ) fz = tanz ( tanz(z) );
      if ( id == 43 ) fz = tanz ( secz(z) );
      if ( id == 44 ) fz = tanz ( cscz(z) );
      if ( id == 45 ) fz = tanz ( cotz(z) );
      if ( id == 46 ) fz = sinhz ( z );
      if ( id == 47 ) fz = coshz ( z );

   #endif

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

void func48 ( int id, vec2 z, inout vec2 fz )
 {
   #ifndef USE_IF_BASED_FUNCTION_SELECTION

     switch ( id - 48 )
      {
        case  0 : fz = tanhz ( z );                                             break;
        case  1 : fz = sechz ( z );                                             break;
        case  2 : fz = cschz ( z );                                             break;
        case  3 : fz = cothz ( z );                                             break;
        case  4 : fz = divz  ( tanz(exp2z(z)), z );                             break;
        case  5 : fz = sinz  ( cosz(sinz(z)) );                                 break;
        case  6 : fz = epowz ( invz(sqrtz(-z)) );                               break;
        case  7 : fz = epowz ( sinz(epowz(cosz(z))) );                          break;
        case  8 : fz = divz  ( sinz(z), z );                                    break;
        case  9 : fz = divz  ( sinz(z), cosz(exp2z(z)) );                       break;
        case 10 : fz = divz  ( sqrtz ( z + z_ONE ), sqrtz ( z - z_ONE ) );      break;
        case 11 : fz = invz  ( z_ONE + mulz(z,exp2z(exp2z(z))) );               break;
        case 12 : fz = sqrtz ( divz ( logz(iz(z)-z_SIX), logz(iz(z)+z_TWO) ) ); break;
        case 13 : fz = sin(z) * sin(z);                                         break;
        case 14 : fz = sin(z) * cos(z);                                         break;
        case 15 : fz = sin(z) * tan(z);                                         break;
      }

   #else

      if ( id == 48 ) fz = tanhz ( z );
      if ( id == 49 ) fz = sechz ( z );
      if ( id == 50 ) fz = cschz ( z );
      if ( id == 51 ) fz = cothz ( z );
      if ( id == 52 ) fz = divz  ( tanz(exp2z(z)), z );
      if ( id == 53 ) fz = sinz  ( cosz(sinz(z)) );
      if ( id == 54 ) fz = epowz ( invz(sqrtz(-z)) );
      if ( id == 55 ) fz = epowz ( sinz(epowz(cosz(z))) );
      if ( id == 56 ) fz = divz  ( sinz(z), z );
      if ( id == 57 ) fz = divz  ( sinz(z), cosz(exp2z(z)) );
      if ( id == 58 ) fz = divz  ( sqrtz ( z + z_ONE ), sqrtz ( z - z_ONE ) );
      if ( id == 59 ) fz = invz  ( z_ONE + mulz(z,exp2z(exp2z(z))) );
      if ( id == 60 ) fz = sqrtz ( divz ( logz(iz(z)-z_SIX), logz(iz(z)+z_TWO) ) );
      if ( id == 61 ) fz = sin(z) * sin(z);
      if ( id == 62 ) fz = sin(z) * cos(z);
      if ( id == 63 ) fz = sin(z) * tan(z);

   #endif

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

void func64 ( int id, vec2 z, inout vec2 fz )
 {
   #ifndef USE_IF_BASED_FUNCTION_SELECTION

      switch ( id - 64 )
       {
         case  0 : fz = sin(z) * sinz(z);                                                break;
         case  1 : fz = sin(z) * cosz(z);                                                break;
         case  2 : fz = sin(z) * tanz(z);                                                break;
         case  3 : fz = sin(z) * secz(z);                                                break;
         case  4 : fz = sin(z) * sinhz(z);                                               break;
         case  5 : fz = sin(z) * coshz(z);                                               break;
         case  6 : fz = sin(z) * tanhz(z);                                               break;
         case  7 : fz = sin(z) * sechz(z);                                               break;
         case  8 : fz = z * epowz(z_i*z.x);                                              break;
         case  9 : fz = z * epowz(z_i*z.y);                                              break;
         case 10 : fz = z * epowz(z_i*(z.x+z.y));                                        break;
         case 11 : fz = exp(z.x)*cosz(z);                                                break;
         case 12 : fz = exp(z.x)*sinz(z);                                                break;
         case 13 : fz = mulz(z,z,z) + z_ONE;                                             break;
         case 14 : fz = divz ( mulz(z-z_ONE,z+z_ONE,z+z_ONE), mulz(z+z_i,z-z_i,z-z_i) ); break;
         case 15 : fz = mulz ( z/4.0, sinz(invz(z/4.0)) );                               break;
       }

   #else

      if ( id == 64 ) fz = sin(z) * sinz(z);
      if ( id == 65 ) fz = sin(z) * cosz(z);
      if ( id == 66 ) fz = sin(z) * tanz(z);
      if ( id == 67 ) fz = sin(z) * secz(z);
      if ( id == 68 ) fz = sin(z) * sinhz(z);
      if ( id == 69 ) fz = sin(z) * coshz(z);
      if ( id == 70 ) fz = sin(z) * tanhz(z);
      if ( id == 71 ) fz = sin(z) * sechz(z);
      if ( id == 72 ) fz = z * epowz(z_i*z.x);
      if ( id == 73 ) fz = z * epowz(z_i*z.y);
      if ( id == 74 ) fz = z * epowz(z_i*(z.x+z.y));
      if ( id == 75 ) fz = exp(z.x)*cosz(z);
      if ( id == 76 ) fz = exp(z.x)*sinz(z);
      if ( id == 77 ) fz = mulz(z,z,z) + z_ONE;
      if ( id == 78 ) fz = divz ( mulz(z-z_ONE,z+z_ONE,z+z_ONE), mulz(z+z_i,z-z_i,z-z_i) );
      if ( id == 79 ) fz = mulz ( z/4.0, sinz(invz(z/4.0)) );

   #endif

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

void func80 ( int id, vec2 z, inout vec2 fz )
 {
   #ifndef USE_IF_BASED_FUNCTION_SELECTION

      switch ( id - 80 )
       {                                                                          break;
         case  0 : fz = mulz ( z, mulz(z,z), mulz(z,z) ) - z_ONE;                 break;
         case  1 : fz = invz ( mulz ( z, mulz(z,z), mulz(z,z) ) ) - z_ONE;        break;
         case  2 : fz = ( z + invz(z) ) / 2.0;                                    break;
         case  3 : fz = mulz  ( sqrtz ( z + z_ONE ), sqrtz ( z - z_ONE ) );       break;
         case  4 : fz = divz ( cosz(z), sinz(mulz(mulz(z,z),mulz(z,z))) );        break;
         case  5 : fz = sum_of_negative_powers(z,5);                              break;
         case  6 : fz = invz ( sum_of_negative_powers(z,5) );                     break;
         case  7 : fz = sinz(invz(z));                                            break;
         case  8 : fz = (z_ONE+z_i)*sinz(z);                                      break;
         case  9 : fz = sinz(mulz(z,z,z)) - z_ONE;                                break;
         case 10 : fz = divz ( sinz(mulz(z,z,z)) - z_ONE, z );                    break;
         case 11 : fz = divz ( z - z_ONE, mulz(z,z) + z + z_ONE );                break;
         case 12 : fz = divz ( cosz(z), sinz(mulz(mulz(z,z),mulz(z,z))-z_ONE ) ); break;
         case 13 : fz = logz ( sinz(z) );                                         break;
         case 14 : fz = logz ( sinhz(z) );                                        break;
         case 15 : fz = logz ( tanhz(z) );                                        break;
       }

   #else

      if ( id == 80 ) fz = mulz ( z, mulz(z,z), mulz(z,z) ) - z_ONE;
      if ( id == 81 ) fz = invz ( mulz ( z, mulz(z,z), mulz(z,z) ) ) - z_ONE;
      if ( id == 82 ) fz = ( z + invz(z) ) / 2.0;
      if ( id == 83 ) fz = mulz  ( sqrtz ( z + z_ONE ), sqrtz ( z - z_ONE ) );
      if ( id == 84 ) fz = divz ( cosz(z), sinz(mulz(mulz(z,z),mulz(z,z))) );
      if ( id == 85 ) fz = sum_of_negative_powers(z,5);
      if ( id == 86 ) fz = invz ( sum_of_negative_powers(z,5) );
      if ( id == 87 ) fz = sinz(invz(z));
      if ( id == 88 ) fz = (z_ONE+z_i)*sinz(z);
      if ( id == 89 ) fz = sinz(mulz(z,z,z)) - z_ONE;
      if ( id == 90 ) fz = divz ( sinz(mulz(z,z,z)) - z_ONE, z );
      if ( id == 91 ) fz = divz ( z - z_ONE, mulz(z,z) + z + z_ONE );
      if ( id == 92 ) fz = divz ( cosz(z), sinz(mulz(mulz(z,z),mulz(z,z))-z_ONE ) );
      if ( id == 93 ) fz = logz ( sinz(z) );
      if ( id == 94 ) fz = logz ( sinhz(z) );
      if ( id == 95 ) fz = logz ( tanhz(z) );

   #endif

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

void func96 ( int id, vec2 z, inout vec2 fz )
 {
   #ifndef USE_IF_BASED_FUNCTION_SELECTION

      switch ( id - 96 )
       {
         case 0 : fz = (z_ONE+z_i) * logz(sinz(divz(mulz(z,z,z)-z_ONE,z))); break;
         case 1 : fz = epowz ( divz ( z - z_ONE, cosz(z) ) );               break;
         case 2 : fz = epowz ( divz ( z - z_ONE, tanz(z) ) );               break;
         case 3 : fz = divz  ( z, tanz(exp2z(z)) );                         break;
       }

   #else

      if ( id == 96 ) fz = (z_ONE+z_i) * logz(sinz(divz(mulz(z,z,z)-z_ONE,z)));
      if ( id == 97 ) fz = epowz ( divz ( z - z_ONE, cosz(z) ) );
      if ( id == 98 ) fz = epowz ( divz ( z - z_ONE, tanz(z) ) );
      if ( id == 99 ) fz = divz  ( z, tanz(exp2z(z)) );

   #endif

 }

#endif // staticFuncId


// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

vec2 evaluate_complex_function ( int id, vec2 z )
 {
   vec2 zz = logOfArgument  ? logz(z)  : z;
        zz = invertArgument ? invz(zz) : zz;

   vec2 fz;

   #if defined staticFuncId

      fz = EVALUATE_FUNCTION ( zz );

   #else

      #ifndef USE_IF_BASED_FUNCTION_SELECTION
         switch ( id >> 4 )
          {
            case 0: func00 ( id, zz, fz ); break;
            case 1: func16 ( id, zz, fz ); break;
            case 2: func32 ( id, zz, fz ); break;
            case 3: func48 ( id, zz, fz ); break;
            case 4: func64 ( id, zz, fz ); break;
            case 5: func80 ( id, zz, fz ); break;
            case 6: func96 ( id, zz, fz ); break;
            // Extend if more functions are supported
          }
      #else
         int xid = id >> 4;
         if ( xid == 0 ) func00 ( id, zz, fz );
         if ( xid == 1 ) func16 ( id, zz, fz );
         if ( xid == 2 ) func32 ( id, zz, fz );
         if ( xid == 3 ) func48 ( id, zz, fz );
         if ( xid == 4 ) func64 ( id, zz, fz );
         if ( xid == 5 ) func80 ( id, zz, fz );
         if ( xid == 6 ) func96 ( id, zz, fz );
         // Extend if more functions are supported
      #endif

   #endif

   fz = logOfFunction  ? logz(fz) : fz;
   fz = invertFunction ? invz(fz) : fz;

   return fz;

 }

// -----------------------------------------------------------------------------

// Convert from cartesian to polar representation.

#define polar(xy) vec2 ( length(xy), atan(xy.y,xy.x) / two_pi )

// -----------------------------------------------------------------------------

// The default colouring is based on the value that has been evaluated for the
// currently selected complex function. It uses two colouring schemes at once.

// The  first is based on the r * exp(i*theta) representation of  the  complex
// value z and, by default, produces a rainbow as a function of theta with the
// Red, Yellow, Green, Cyan, Blue and Magenta segments being centered at theta
// corresponding to 0, 60, 120, 180, 240 and 300 degrees repectively. A set of
// whilte lines is suuperimposed which are,  by default,  centered  on integer
// values of r. The default spacing of the colour bands and white lines can be
// modified as described in the comments at the start of this source file.
//
// The second colouring used is based on the cartesian representation  x + i*y
// of z and, by default, produces dark lines at integer values of x and y. The
// default spacing of the dark lines can be modified as described by  comments
// at the start of this source file.
//
// The sets of dark and white lines may be regarded as x, y and r contours. By
// default no theta contours are drawn as they can tend to wash out the colour
// bands, however they can be enabled via the contour control parameters as is
// described in the comments at the start of this source file.
//
// Technicaly the colouring is performed by working in the HSV (hue,saturation
// and value) colour system with theta selection the basic hue which then  has
// its saturation (darkness) modulated by the distance from (x,y) contours and
// its value (whiteness) component is modulated by the distance from (r,theta)
// contours. By default the modulation produces rather fuzzy contours, this is
// deliberately so for artistic reasons though their widths do indicate to the
// rate of change of the function with broad areas indicating flat regions  of
// the co-domain. The sharpness of the contours may be altered as is described
// in the comments at the start of this source file.
//
// The arguments z and p should normaly be the cartesian and equivalent  polar
// representations of the same complex number,  but for an alternate colouring
// scheme this rule can be broken.  If it is broken then the interpretation of
// the relationship between the two sets of contours is more complicated  than
// when the nominal relationship between the arguments holds.
//
// This function is based on Huw Bowles version but with minor enhancements.

void modify_according_to_contour_mode ( int mode, inout float q, inout float f  )
 {
   switch ( mode )
    { case 0 : q = atan(q); q = 4.0*abs(q)*q; break;
      case 1 :                                break;
      case 2 : q = sign(q)*log(abs(q));       break;
      case 3 : q = atan(q);                   break;
      case 8 : q = atan(q); f = 1.0-abs(q);   break;
      case 9 : f = 0.0;                       break;
    }
 }

const vec3 HSV_to_RGB_K = vec3(0.0,4.0/6.0,2.0/6.0);

vec4 simpleColouring ( vec2 z, vec2 p, vec4 fade )
 {
   // Determine the colour based on the theta part of the
   // polar representation of z where z = r * i*theta.

   float sat = 1.0;
   float val = 0.5;

   if ( ContourMode != 99 )
    {
      // Determine modulation factors for x and y contours as xy_ra.xy and
      // determine modulation factors for r and theta contours as xy_ra.zw.

      vec4 xy_ra = vec4(z,p);
      xy_ra = sign(xy_ra)*pow(abs(xy_ra),ContourPowerFactor) * ContourScale;

      xy_ra = abs ( fract(xy_ra) - 0.5 ) * ContourIntensity * fade; // ContourIntensity is 0 or 2
      xy_ra = pow ( abs(xy_ra), ContourSharp );
      xy_ra = max ( 1.0-xy_ra, 0.0 );

      // Apply the contour modulation factors.

      sat = xy_ra.z * xy_ra.w;
      val = xy_ra.x * xy_ra.y;

      val = mix ( 1.0, val, sat*0.5 );
    }

   float hue = ( p.y * ColourScaleFactor ); // N.B. the angle is in "turns".

   // Convert from HSV colour representation to RGB.

   vec3 rgb = abs ( fract(hue+HSV_to_RGB_K)*6.0 - 3.0 ) - 1.0;
   rgb = clamp ( rgb, 0.0, 1.0 );

   return vec4 ( val * mix ( vec3(1.0), rgb, sat ), 1.0 );

 }

// -----------------------------------------------------------------------------

// As an alternative to the default simple colouring scheme the image may be
// coloured by using the function's components to index into a texture. This
// will result in the final image comprising a tiling of distorted copies of
// the texture image. As a simple tiling would leave discontinuities at  the
// edges of each tile (unless the tiles were designed to avoid it) alternate
// tiles are optionaly flipped so that there is continuity of colour  across
// tile boundaries thereby improving the look of the final  image.  Flipping
// is optional because some textures may be deliberately  directional,  e.g.
// have an arrow on them, and it is desired to maintain this directionality.

#define texturedColours1(z) texture2D ( ColouringTexture, fract(z) )

vec4 texturedColours2 ( vec2 z )
 {

   // Fold about the x=1.0 and y=1.0 to ensure  continuity  of
   // colour across the edges of the squares tiling the image.

   z = abs(z);
   z = mix ( z, -z, float(lessThan(z,vec2_1)) );

   return texturedColours1(z);

 }

// End of code dependant on EVALUATE_FUNCTION

#endif // EVALUATE_FUNCTION

// -----------------------------------------------------------------------------
// -- EMU_DATA_PANEL support ---------------------------------------------------
// -----------------------------------------------------------------------------

#ifdef USE_EMU_DATA_PANEL

// -----------------------------------------------------------------------------

// A very crude way of displaying numerical values using a very ugly font. The
// image for each of the characters is an of 5 lines of 6 dots encoded as a 30
// bit integer. This is expanded to a larger character size by simply  scaling
// each of the encoded dots to larger,  rectangular,  dots in the final image.
// Each character is positioned at the center of a slighly larger cell so that
// the characters do not touch each other.  The result is not very pretty, but
// is not too bad and is useful for debugging.

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Internal parameters and types.

const ivec2 emu_data_panel_char_base_size = ivec2(6,5);

// The primary data panel parameters - these may be set by the user.

struct emu_data_panel_parameters
 {
   vec2  scale;

   float pad_under;
   float pad_over;
   float pad_left;
   float pad_right;

   float line_under;
   float line_over;
   float line_edge;
 };

// Secondary data panel parameters - these are derived from the primary
// data panel parameters and must not be modified by the user. They are
// used to avoid having to recalculate various quantities.

struct emu_data_panel_derived_parameters
 { vec4  colour;
   vec2  inverse_scale;
   vec2  inverse_size;
   vec2  pad;
   vec2  cell;
   vec2  shim;
   vec2  edge;
   vec2  base;
   float over;
   float under;
 };

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Each character requires 30 bits to encode it, 6 bits for each of the
// five lines.  It  is  therefore convenient to use octal when defining
// the characters as each octal digit corresponds to 3 bit so  each  of
// the 5 lines is encoded using a pair of octal digits. As integers are
// 32 bits in GLSL there are two free bits that could be used for flags
// of some kind, but they are currently not used.
//
// The GLSL shader language uses the same horrible convention as C  for
// octal, i.e. a leading 0 means that the number is in octal, yeuk, but
// so be it.  This  this data really should to be declared as constant,
// but constant arrays are not supported by the language.

uint emu_data_panel_alphameric_chars[36]
 = uint[36] ( 007763636377u, // 0
              000636060606u, // 1
              007703776077u, // 2
              007703770377u, // 3
              006363770303u, // 4
              007760770377u, // 5
              007760776377u, // 6
              007706143060u, // 7
              007763776377u, // 8
              007763770303u, // 9
              003663776363u, // A
              007663766376u, // B
              007763606377u, // C
              007663636376u, // D
              007760776077u, // E
              007760776060u, // F
              003760676337u, // G
              006363776363u, // H
              007714141477u, // I
              007714145474u, // J
              006366706663u, // K
              006060606077u, // L
              004163554141u, // M
              006151454341u, // N
              003663636336u, // O
              007663766060u, // P
              003641454335u, // Q
              007663765446u, // R
              003760360376u, // S
              007714141414u, // T
              006363636336u, // U
              004141412214u, // V
              004141556341u, // W
              004122142241u, // X
              004141221414u, // Y
              007706143077u  // Z
           );

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// The overall panel record.  The  user must declare and initialise one
// of these for each panel that is to be used and may directly set  the
// values used for the colour and params components but must not change
// the values of the other components.
//
// After any params changes  emu_data_panel_evaluate_derived_parameters
// must be called to keep the params and params2 components consistant.

struct emu_data_panel
 {
   vec4[3] colour;                   // User read and write access OK.
   emu_data_panel_parameters params; // User read and write access OK.

   vec2 cursor; // User may read the cursor, but not write to it.

   bool done;   // Flags that the pixel has been processed.

   emu_data_panel_derived_parameters params2; // No user access allowed.

 };

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Evaluate a panel's derived parameters.  This must be called after
// any change is made to any of the panel's params field's elements.

void emu_data_panel_evaluate_derived_parameters ( inout emu_data_panel p )
 {
   p.done = false;

   p.params2.colour = vec4(0.0);

   p.params2.inverse_scale
     = 1.0 / p.params.scale;

   p.params2.inverse_size
     = p.params2.inverse_scale / vec2(emu_data_panel_char_base_size);

   p.params2.pad
     = vec2 ( p.params.pad_left+p.params.pad_right,
              p.params.pad_under+p.params.pad_over
            );

   p.params2.cell
     = ( vec2(emu_data_panel_char_base_size)
       + vec2(p.params2.pad)
       + vec2(0.0,p.params.line_under+p.params.line_over)
       ) * p.params.scale;

   p.params2.shim
     = vec2(p.params.scale.x,p.params2.cell.y);

   p.params2.edge
     = vec2 ( p.params.line_edge*p.params.scale.x, p.params2.cell.y );

   p.params2.base
     = vec2 ( p.params.pad_left,
              p.params.pad_under + p.params.line_under
            ) * p.params.scale;

   p.params2.over
     = emu_data_panel_char_base_size.y + p.params.pad_over;

   p.params2.under
    = - p.params.pad_under;

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Get a panel's pixel colour. Use this if the panel is to be opaque.

#define emu_data_panel_pixel_colour(p) p.params2.colour

// Merge a panel with whatever image it is to overlay.  The  degree of
// blending is controlled by the alpha component of the panel's colour
// which specifies how opaque the panel colour is to be. However, this
// may be modified by periodicaly fading the panel in and out.

vec4 emu_data_panel_merge ( emu_data_panel p, vec4 background )
 {
   float aa = p.params2.colour.a;

   #ifdef USE_EMU_DATA_PANEL_CYCLE

      if ( DataPanelCycle.x > 0.0 )
       {
         float tt = mod ( Time, DataPanelCycle.x );
         float t1 = DataPanelCycle.y;      // Time to start to fade in
         float t2 = t1 + DataPanelCycle.w; // plus the fade's duration
         float t3 = t1 + DataPanelCycle.z; // Time to start to fade out
         float t4 = t3 + DataPanelCycle.w; // plus the fade's duration

         float a1 = smoothstep ( t1, t2, tt );
         float a2 = smoothstep ( t3, t4, tt );

         aa *= a1 * (1.0-a2);
       }

   #endif

   vec4 result;

   result.rgb = mix ( background.rgb, p.params2.colour.rgb, aa );
   result.a   = max ( background.a, aa );

   return result;
 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Explicit horizontal and vertical cursor movements.  The full forms,
// i.e. those with two arguments, move the cursor by the fraction f of
// a character cell. Normaly the single argument forms move it by just
// a single cell's width or height.

void emu_data_panel_cursor_left ( inout emu_data_panel p, float f )
 { p.cursor.x -= p.params2.cell.x*f;
 }

void emu_data_panel_cursor_right ( inout emu_data_panel p, float f )
 { p.cursor.x += p.params2.cell.x*f;
   p.done = false;
 }

void emu_data_panel_cursor_up ( inout emu_data_panel p, float f )
 { p.cursor.y += p.params2.cell.y*f;
   p.done = false;
 }

void emu_data_panel_cursor_down ( inout emu_data_panel p, float f )
 { p.cursor.y -= p.params2.cell.y*f;
   p.done = false;
 }

void emu_data_panel_cursor_left ( inout emu_data_panel p ) { p.cursor.x -= p.params2.cell.x; }
void emu_data_panel_cursor_right( inout emu_data_panel p ) { p.cursor.x += p.params2.cell.x; }
void emu_data_panel_cursor_up   ( inout emu_data_panel p ) { p.cursor.y += p.params2.cell.y; }
void emu_data_panel_cursor_down ( inout emu_data_panel p ) { p.cursor.y -= p.params2.cell.y; }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Move a panel's cursor to the left by the space required for  what
// is being output check if the current pixel is in the area that is
// to receive whatever it is that is being output.

bool emu_data_panel_update_and_check_cursor ( inout emu_data_panel p, vec2 xy )
 {
   p.cursor.x -= xy.x;

   return all ( lessThan         ( gl_FragCoord.xy, p.cursor + xy ) )
       && all ( greaterThanEqual ( gl_FragCoord.xy, p.cursor      ) );

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// emu_data_panel_put_char - outputs a single character, e.g. a digit.
// The panel's cursor is moved left by one character cell's  width  to
// accomodate the new character which is output to the  right  of  the
// updated cursor.
//
// emu_data_panel_put_char may be called by the user, but normally not
// as the user should use the higher level macros or functions such as
// emu_data_panel_put_int.

void emu_data_panel_put_char ( inout emu_data_panel p, uint c )
 {

   if ( (!p.done) && emu_data_panel_update_and_check_cursor(p,p.params2.cell) )
    {
      // Get position of the current pixel realtive to the
      // newly updated position of the cursor.

      vec2 q = ( gl_FragCoord.xy - p.params2.base - p.cursor );

      // Convert q to an index within the 3x5 character. This is
      // in the range 0.0 to 1.0 if q is in this area  but  will
      // be outside this range if q does not address  this  part
      // of the character cell

      vec2 a = q * p.params2.inverse_size;

      uint k = 0u; // Assume background colour.

      if ( a == fract(a) )
       { // a addresses the central 6x5 pixel character. Convert
         // it into a bit number within c, extract that bit from
         // c and use it as k to select either the background or
         // the foreground colour.
         k = uint(floor(5.0*a.y))*6u + 5u - uint(floor(6.0*a.x));
         k = ( c >> k ) & 1u;
       }
      else
       { // Not the central 6x5 pixels. Set k to 2 if we are in
         // the optional edge region otherwise leave it as 0.
         float b = q.y * p.params2.inverse_scale.y;
         if ( ( b <= p.params2.under ) || ( b >= p.params2.over ) )
          { k = 2u;
          }
       }

      // Set the colour of the current pixel.

      p.params2.colour = p.colour[k];

      // Indicate that the current pixel has been processed.

      p.done = true;

    }
 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

void emu_data_panel_put_letter ( inout emu_data_panel p, uint c )
 { emu_data_panel_put_char ( p, emu_data_panel_alphameric_chars[uint(c)+9u] );
 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// A few extra characters are also defined.

void emu_data_panel_put_char_blank   ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 000000000000u ); }
void emu_data_panel_put_char_minus   ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 000000770000u ); }
void emu_data_panel_put_char_plus    ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 000014771400u ); }
void emu_data_panel_put_char_dot     ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 000000140000u ); }
void emu_data_panel_put_char_block   ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 007777777777u ); }
void emu_data_panel_put_char_dotdot  ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 000000630000u ); }
void emu_data_panel_put_char_dash    ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 000000360000u ); }
void emu_data_panel_put_char_colon   ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 000014001400u ); }
void emu_data_panel_put_char_vline1  ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 001414141414u ); }
void emu_data_panel_put_char_vline2  ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 001414001414u ); }
void emu_data_panel_put_char_vline3  ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 001400140014u ); }
void emu_data_panel_put_char_2vlines1( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 000022222200u ); }
void emu_data_panel_put_char_2vlines2( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 002222222222u ); }
void emu_data_panel_put_char_exclam  ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 001414140014u ); }
void emu_data_panel_put_char_slash   ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 000306143060u ); }
void emu_data_panel_put_char_bslash  ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 006030140603u ); }
void emu_data_panel_put_char_times   ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 000063146300u ); }
void emu_data_panel_put_char_equals  ( inout emu_data_panel p ) { emu_data_panel_put_char ( p, 000036003600u ); }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// emu_data_panel_put_shim - outputs a "shim" of the background  colour
// with width equal to one character pixel times the second argument f.
// The underline and overline options for the panel are obeyed so those
// parts of a shim will be rendered using the edge colour.

void emu_data_panel_put_shim ( inout emu_data_panel p, float f )
 {

   if ( (!p.done) && emu_data_panel_update_and_check_cursor ( p, p.params2.shim*vec2(f,1.0)) )
    {
      vec2  q = ( gl_FragCoord.xy - p.params2.base - p.cursor );
      float b = q.y * p.params2.inverse_scale.y;

      int k = 0;

      if ( ( b <= p.params2.under ) || ( b >= p.params2.over ) )
       { k = 2;
       }

      p.params2.colour = p.colour[k];
      p.done = true;

    }
 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// emu_data_panel_align_right - align the right hand edges of two panels.

void emu_data_panel_align_right ( inout emu_data_panel p1, inout emu_data_panel p2 )
 { float r0 = min ( p1.cursor.x, p2.cursor.x );
   float r1 = max(p1.cursor.x-r0,0.0) * p1.params2.inverse_scale.x;
   float r2 = max(p2.cursor.x-r0,0.0) * p2.params2.inverse_scale.x;
   if ( r1 > 0.0 ) emu_data_panel_put_shim ( p1, r1 );
   if ( r2 > 0.0 ) emu_data_panel_put_shim ( p2, r2 );
 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// emu_data_panel_put_edge - outputs a vertical panel edge. The cursor
// is moved left by the edge thickness to accomodate the edge which is
// a vertical line the full character cell height of the edge colour.

void emu_data_panel_put_edge ( inout emu_data_panel p )
 {
   if ( (!p.done) && emu_data_panel_update_and_check_cursor(p,p.params2.edge) )
    { p.params2.colour = p.colour[2];
      p.done = true;
    }

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// emu_data_panel_put_decimal_digit - outputs a single decimal digit.
// emu_data_panel_put_hex_digit - outputs a single hexadecimal digit.

// The user may call these, but normally should have no need to do so.

void emu_data_panel_put_decimal_digit ( inout emu_data_panel p, uint d )
 { emu_data_panel_put_char ( p, emu_data_panel_alphameric_chars[d-(d/10u)*10u] );
 }

void emu_data_panel_put_hex_digit ( inout emu_data_panel p, uint d )
 { emu_data_panel_put_char ( p, emu_data_panel_alphameric_chars[d & 0xfu] );
 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// put sign = outputs the sign of v.

// The user may call this, but normally should have no need to do so.

void emu_data_panel_put_sign ( inout emu_data_panel p, int v, int s )
 {
   if ( v<0 )
    {
      emu_data_panel_put_char_minus ( p );
    }
   else
    {
      if ( s == 1 ) emu_data_panel_put_char_blank ( p );
      if ( s == 2 ) emu_data_panel_put_char_plus  ( p );
    }
 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// emu_data_panel_put_uint - outputs an unsigned decimal integer.
// emu_data_panel_put_int  - outputs a decimal integer.
//
// The argument s for emu_data_panel_put_int specifies the style to
// be used for non-negative values.
//
//    s = 0 - no output when v is not negative
//    s = 1 - use blank when v is not negative
//    s = 2 - plus sign when v is not negative
//
// a minus sign will always be output for negative values.

void emu_data_panel_put_uint ( inout emu_data_panel p, uint v  )
 {
   if ( !p.done ) do
    { emu_data_panel_put_decimal_digit ( p, v );
      v /= 10u;
    } while ( v > 0u );
 }

void emu_data_panel_put_uint ( inout emu_data_panel p, uint v, int w  )
 {
   if ( !p.done ) do
    { emu_data_panel_put_decimal_digit ( p, v );
      v /= 10u;
      w -= 1;
    } while ( v != 0u || w > 0 );
 }

void emu_data_panel_put_int ( inout emu_data_panel p, int v, int s  )
 {
   if ( !p.done )
    { emu_data_panel_put_uint ( p, uint(abs(v)) );
      emu_data_panel_put_sign ( p, v, s );
    }
 }

void emu_data_panel_put_int ( inout emu_data_panel p, int v )
 {
   emu_data_panel_put_int ( p, v, 0 );
 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Support for outputing hexadecimal integers is only enabled if it is
// explictly enabled by defining USE_EMU_DATA_PANEL_PUT_HEX.

#ifdef USE_EMU_DATA_PANEL_PUT_HEX

// emu_data_panel_put_hex - outputs a hexadecimal integer. By default
// there are no leading zeros (or any other leading character) but if
// the optional minimal width argument,  w,  then  then leading zeros
// are used if they are required.

void emu_data_panel_put_hex ( inout emu_data_panel p, uint v )
 {
   uint z = ~0u;
   do
    { emu_data_panel_put_hex_digit ( p, v );
      v >>= 4;
      z <<= 4;
    } while ( z != 0u );
 }

void emu_data_panel_put_hex ( inout emu_data_panel p, uint v, int w )
 {
   do
    { emu_data_panel_put_hex_digit ( p, v );
      w -= 1;
      v >>= 4;
    } while ( v != 0u || w > 0 );
 }

void emu_data_panel_put_hex ( inout emu_data_panel p, int v )
 { emu_data_panel_put_hex ( p, uint(v) );
 }

void emu_data_panel_put_hex ( inout emu_data_panel p, int v, int w )
 { emu_data_panel_put_hex ( p, uint(v), w );
 }

#endif // USE_EMU_DATA_PANEL_PUT_HEX

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Support for outputing floating point values is only enabled if it
// is explictly enabled by defining USE_EMU_DATA_PANEL_PUT_FLOAT.

#ifdef USE_EMU_DATA_PANEL_PUT_FLOAT

void do_emu_data_panel_put_float ( inout emu_data_panel p, float v, int fore, int aft )
 {
   fore = clamp ( fore, 0, 9 );
   aft  = clamp ( aft,  0, 9 );

   if ( IS_NAN(v) || ( v == v*2.0 + 1.0 ) ) // test for NaN or inf
    {
      int nn = aft+fore;
      for ( int i=0; i<nn i++ )
       { if ( i == aft ) emu_data_panel_put_char_dot ( p );
         emu_data_panel_put_char_block ( p );
       }
      return;
    }

   float fff = pow(10.0,fore);

   float a = abs ( v );

   int e = 0;

   if ( a != 0.0 )
    { while ( a > fff ) { a /= 10.0; e++; }
      while ( a < 0.1 ) { a *= 10.0; e--; }
    }

   float c; float b = modf(a,c);

   if ( e != 0 )
    { emu_data_panel_put_int ( p, e );
      emu_data_panel_put_char_E ( p );
    }

   emu_data_panel_put_uint ( p, uint(b*pow(10.0,aft)), aft );
   emu_data_panel_put_char_dot ( p );
   emu_data_panel_put_int ( p, int(c*sign(v)) );

 }

void emu_data_panel_put_float ( inout emu_data_panel p, float v, int fore, int aft )
 {
   if ( !p.done ) do_emu_data_panel_put_float ( p, v, fore, aft );
 }
#endif // USE_EMU_DATA_PANEL_PUT_FLOAT

// -----------------------------------------------------------------------------

// Output "debugging" information.  Note,  because  of the way the underlying
// debugging utilities have been written, these data items have to be written
// in reverse order.  The data will be displayed as up to three sub-panels in
// the bottom left of the screen. From left to right these sub-panels show:-
//
//    The current function id
//    The minimum function id - Only present when the
//    The maximum function id - function id is varying
//
//    The current iteration level
//    The minimum iteration level - Only present when the
//    The maximum iteration level - min and max differ.
//
//    The function mode being used
//    The colour scheme being used
//
// The central panel giving the iteration data is only output when a function
// is being iteratated.
//
// This display enables the user to determine the parameter values that  were
// used for a particular image when the shader is progressing through a  long
// sequence of them. It is also useful for debugging.

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Provide a basic panel definition. In principle this is tailored to
// the particular shader, though in this case it is general enough to
// be used for many purposes.

emu_data_panel complex_functions_data_panel ( void )
 {
   // Local variable in which the panel definition will built
   // up. Its value will be returned as tat of the function.

   emu_data_panel result;

   // Define the position of the panel's bottom right corner.
   // In this case its at the bottom right of the window.

   result.cursor
     = vec2 ( WindowSize.x, 0.0 ); // Bottom right corner.

   // Define the colours to be used for the panel.

   result.colour
     = vec4[3] ( vec4 ( 0.0, 0.0, 0.0, 0.3 ), // Background colour.
                 vec4 ( 1.0, 1.0, 1.0, 0.9 ), // Foreground colour.
                 vec4 ( 0.7, 0.7, 0.7, 0.9 )  // Border colour.
               );

   // Define the ramining panel parameters. You can either
   // do them one by one or, as here, all at once.

   result.params
       = emu_data_panel_parameters
          (
            /* scale      */ vec2(1.0,3.0),
            /*            */
            /* pad_under  */ 0.5,
            /* pad_over   */ 0.5,
            /* pad_left   */ 0.5,
            /* pad_right  */ 0.5,
            /*            */
            /* line_under */ 0.5,
            /* line_over  */ 0.5,
            /* line_edge  */ 1.0
          );

   emu_data_panel_evaluate_derived_parameters ( result );

   return result;

 }

// End of shader independant code dependant on USE_EMU_DATA_PANEL.

#endif // USE_EMU_DATA_PANEL

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// The following code using the data panel support above is specific to
// this particular shader and has to be re-written foe each shader that
// makes use of the data panel library.

#ifdef USE_EMU_DATA_PANEL

// Shader specific function to the data panel.

vec4 Data_Panel_Contents ( vec4 background )
 {
   emu_data_panel pp = complex_functions_data_panel();

   // The panel's parameters may be modified here,  but any changes
   // must be followed by a call to
   //
   //      emu_data_panel_evaluate_derived_parameters
   //
   // to update its internal state.

   // For this panel emu_data_panel_put_uint has been used rather
   // than emu_data_panel_put_int because we want to always use 2
   // digits, with leading zeros if required, for numbers and the
   // current version of emu_data_panel_put_int doesn't provide a
   // simple way to output signed ints with a fixed width.

   // Optionaly provide the colouring and transition mode data.

   #ifdef USE_EMU_DATA_PANEL_PART_3
      emu_data_panel_put_edge ( pp );
      emu_data_panel_put_uint ( pp, uint(TransitionMode), 2 );
      emu_data_panel_put_char_dot ( pp );
      emu_data_panel_put_uint ( pp, uint(ContourMode), 2 );
      emu_data_panel_put_char_dot ( pp );
      emu_data_panel_put_uint ( pp, uint(ColourScheme), 2 );
      emu_data_panel_put_edge ( pp );
      emu_data_panel_cursor_left ( pp, 0.5 );
   #endif

   // Optionaly output the domain mapping and trimming data.

   #ifdef USE_EMU_DATA_PANEL_PART_2
      emu_data_panel_put_edge ( pp );
      emu_data_panel_put_uint ( pp, uint(DomainTrimMode), 2 );
      emu_data_panel_put_char_dot ( pp );
      emu_data_panel_put_uint ( pp, uint(DomainMapScaleMode), 2 );
      emu_data_panel_put_char_dot ( pp );
      emu_data_panel_put_uint ( pp, uint(DomainMapMode), 2 );
      emu_data_panel_put_edge ( pp );
      emu_data_panel_cursor_left ( pp, 0.5 );
   #endif

   // Optionaly output function mode and iteration data but always
   // output the function id itself and the series letter, if any.

   emu_data_panel_put_edge ( pp );

   #ifdef USE_EMU_DATA_PANEL_PART_1
      emu_data_panel_put_uint ( pp, uint(currentIteration), 2 );
      emu_data_panel_put_char_dot ( pp );
      emu_data_panel_put_uint ( pp, uint(FunctionMode), 2 );
      emu_data_panel_put_char_dot ( pp );
   #endif

   emu_data_panel_put_uint ( pp, uint(functionIdX.y), 2 );
   emu_data_panel_put_edge ( pp );
   emu_data_panel_cursor_left ( pp, 0.5 );

   if ( SeriesLetter > 0 )
    { emu_data_panel_put_edge ( pp );
      uint ss = uint(SeriesLetter);
      uint s1, s2;
      do
       { s1 = ss / 100u;
         emu_data_panel_put_letter ( pp, uint(ss-100u*s1) );
         ss = s1;
       } while ( ss > 0u );
      emu_data_panel_put_edge ( pp );
    }

   return emu_data_panel_merge ( pp, background );

 }

vec4 Data_Panel ( vec4 background )
 {
   // A  quick check that the current pixel could possibly be  part
   // of the panel.  The  panel  height has been hard coded in this
   // check because we do not want to evaluate the height for every
   // pixel. It is crude but this function has to be re-written for
   // every new panel anyway its not too bad.  This  check does not
   // have to be 100% accurate as long as it does not  discard  any
   // pixels that really are part of the panel.

   if ( gl_FragCoord.x < WindowSize.x*0.75 || gl_FragCoord.y > 21.0 )
    {
      return background;
    }
   else
    {
      return Data_Panel_Contents ( background );
    }

 }
// End of shader specific code dependant on USE_EMU_DATA_PANEL.

#endif // USE_EMU_DATA_PANEL

// -----------------------------------------------------------------------------
// -----------------------------------------------------------------------------
// -----------------------------------------------------------------------------

// The main routine converts the pixel position to a complex  number,  z,  with
// z = (0.0,0.0) being at the center of the window. The scaling is such that if
// the window is wider than it is high z = (0,pi) and z = (0,-pi)  will  be  at
// the middle of the top and bottom of the window respectively with  z = (pi,0)
// and z = (-pi,0) both lying within the window.  However, if the window height
// is larger than its width this is modified so that z = (pi,0) and z = (-pi,0)
// still lie within the window and a greater range of the y component o f z  is
// covered. This is done because the (pi,0) and (-pi,0) points are rather  more
// interesting than the (0,pi) and (0,-pi) points for the  functions  that  are
// considered here.
//
// After determining z it is used as the argument  to  the  currently  selected
// function, f, yeilding a second complex number fz = f(z).  This  is subjected
// to a rotation such that if fz evaluates to r*exp(i*theta) then theta becomes
// to theta+ColourPhase where ColourPhase is a linear function of time.
//
// Finaly the function value, fz, is used as the basis for colouring the  final
// image.  This may be based on a texture lookup or may use the default  colour
// scheme.
//
// Note, in general complex functions of complex variable are multi-valued.  In
// this program only the principal values are considered.  Where  this involves
// cuts in the function value these can often be seen as discontinuities in the
// colouring across the cuts, e.g. a jump from red to green. However, depending
// on the function and the use of the ColourScale parameter the colours on both
// sides of a cut can "accidently" match disguising the cut or there may  be  a
// colour discontinuity where there is no cut,  e.g. if ColourScale is set to a
// non integer value.

#ifdef EVALUATE_FUNCTION

vec2 Domain_Value_Scaling ( void )
 {
   switch ( DomainMapScaleMode )
    { case 0 : return vec2 (  pi / min(WindowSize.x*1.3, WindowSize.y) );
      case 1 : return vec2 ( 1.0 / min(WindowSize.x,     WindowSize.y) );
      case 2 : return (pi/1.3) / WindowSize;
      case 3 : return 1.0 / WindowSize;
      case 4 : return DomainTrimParams.zw / min(WindowSize.x,WindowSize.y);
      case 5 : return DomainTrimParams.zw / WindowSize;
    }
 }

vec2 DomainValue_Scale = Domain_Value_Scaling();

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

vec4 Codomain_Colour ( vec2 fz )
 {
   // If some problem occurs when calculating f(z)  then  one  or
   // more of the components of fz could have the special  value,
   // NaN, meaning "not a number".   NaN indicates that something
   // like 0.0 / 0.0 occurred or that a  mathematical  function's
   // argument was out of range.  If a NaN is found the NanColour
   // will be used rather than calculating a colour.

   if ( IS_NAN(fz) ) return NanColour;

   // Get the equivalent polar form of the function's value.

   vec2 pz = polar(fz);

   // Colour the current pixel according to the function's value
   // and the selected colouring scheme. Note, in some cases the
   // roles of fz and pz are swapped and the normal relationship
   // expected between the two parameters of simpleColouring has
   // been deliberately broken.

   vec4 colour = vec4_0;
   vec4 contourFade = exp(Contour_FadeRate*abs(pz.x));

   modify_according_to_contour_mode ( xyContourMode, fz.x, contourFade.x );
   modify_according_to_contour_mode ( xyContourMode, fz.y, contourFade.y );
   modify_according_to_contour_mode ( rContourMode,  pz.x, contourFade.z );

   switch ( ColourScheme )
    { // Simple internal colour schemes.
      case 0 : colour = simpleColouring ( fz, pz, contourFade );        break;
      case 1 : colour = simpleColouring ( pz, polar(pz), contourFade ); break;
      case 2 : colour = simpleColouring ( pz, fz, contourFade );        break;
      // Texture based colour schemes.
      case 10 : colour = texturedColours1 ( fz ); break;
      case 11 : colour = texturedColours2 ( fz ); break;
      case 12 : colour = texturedColours1 ( pz ); break;
      case 13 : colour = texturedColours2 ( pz ); break;
    }

   colour.a = colour.a * (1.0-Translucency);

   return colour;

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

bool Trim ( vec2 z )
 {
   bool trim = false;

   z = z - DomainTrimParams.xy; // Recenter
   z = z / DomainTrimParams.zw; // Rescale
   z = abs(z);

   switch ( DomainTrimMode )
    {
      // No trimming

      case 0 : trim = false; break;

      // Square or rectangular area trimming

      case 1 : trim = any ( greaterThan(z,vec2_1) ); break;
      case 2 : trim = all ( lessThan   (z,vec2_1) ); break;

      // Diamond trimming

      case 3 : trim = ( z.x + z.y ) > 1.0; break;
      case 4 : trim = ( z.x + z.y ) < 1.0; break;

      // Circular or eliptical trimming

      case 5 : trim = length(z)*DomainTrimParams.z*DomainTrimParams.w > 1.0; break;
      case 6 : trim = length(z)*DomainTrimParams.z*DomainTrimParams.w < 1.0; break;

    }

   return trim;

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

void Apply_Domain_Map ( inout vec2 z )
 {
   float lz = length(z);

   z = z * DomainRotation;

   switch ( DomainMapMode )
    { // case 0 : z = z;                                  break;
      case  1 : z = 1.0 / z;                              break;

      case  2 : z = z * lz;                               break;
      case  3 : z = 1.0 / ( z * lz );                     break;

      case  4 : z = z * ( abs(z) / (1.0-abs(z)) );        break;
      case  5 : z = z / ( abs(z) / (1.0-abs(z)) );        break;

      case  6 : z = z * ( lz / (1.0-lz) );                break;
      case  7 : z = z / ( lz / (1.0-lz) );                break;

      case  8 : z = z * tan ( abs(z) * pi_by_2 );         break;
      case  9 : z = z / tan ( abs(z) * pi_by_2 );         break;

      case 10 : z = z * tan ( lz*pi_by_2 );               break;
      case 11 : z = z / tan ( lz*pi_by_2 );               break;

      case 12 : z = z * abs ( abs(z) / (1.0-abs(z)) );    break;
      case 13 : z = z / abs ( abs(z) / (1.0-abs(z)) );    break;

      case 14 : z = z * abs ( lz / (1.0-lz) );            break;
      case 15 : z = z / abs ( lz / (1.0-lz) );            break;

      case 16 : z = z * abs ( tan ( abs(z)*pi_by_2 ) );   break;
      case 17 : z = z / abs ( tan ( abs(z)*pi_by_2 ) );   break;

      case 18 : z = z * abs ( tan ( lz*pi_by_2 ) );       break;
      case 19 : z = z / abs ( tan ( lz*pi_by_2 ) );       break;
    }

   z = z * DomainMapScaling;

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

void Get_Domain_Value ( in vec2 xy, out vec2 z )
 {
   z = ( 2.0*xy - WindowSize ) * DomainValue_Scale;
   z = (z+PreAdd) * DomainScale;
   z = sign(z) * pow ( abs(z), DomainPower );
 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Map the current pixel position to the corresponding complex number.
// At the same we time check if the pixel is to be trimmed or not  and
// only do the mapping if the pixel would not be discarded.

void Map_Window_to_Complex_Domain ( out vec2 z, out bool trim )
 {
   Get_Domain_Value ( gl_FragCoord.xy, z );
   trim = Trim(z);
   if ( ! trim ) { Apply_Domain_Map ( z ); }
 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Evaluate the function with the requested level of iteration.  If the
// function id is invalid z will be left unchanged.  Any final codomain
// rotation or other codomain modifiers are then applied. Note that two
// function values are returned in fz,  the first two are the the value
// evaluated for the previous function or iteration of the function and
// the second two are the value for the current interation.  This is to
// provide the inputs needed for the cross fading between the images of
// the previous and and current functions.

void Map_Domain_to_Codomain ( vec2 z, out vec4 fz )
 {
   fz.xy = z;

   for ( int i=0; i<currentIteration; i++ )
    { fz.xy = evaluate_complex_function ( functionIdX.y, fz.xy );
    }

   fz.zw = evaluate_complex_function ( functionIdX.y, fz.xy );

   fz.zw = fz.zw * ColourPhaseMatrix;
   fz.zw = (fz.zw+PostAdd) * CodomainScale;
   fz.zw = sign(fz.zw) * pow ( abs(fz.zw), CodomainPower );

   if ( Transitioning && currentIteration == 0 )
    { fz.xy = evaluate_complex_function ( functionIdX.x, z );
      fz.xy = fz.xy * ColourPhaseMatrix;
      fz.xy = (fz.xy+PostAdd) * CodomainScale;
      fz.xy = sign(fz.xy) * pow ( abs(fz.xy), CodomainPower );
    }

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

vec4 Main_Image ( void )
 {

   bool trim;

   vec2 zz;
   vec4 fz;

   Map_Window_to_Complex_Domain ( zz, trim );

   if ( trim ) return TrimColour;

   Map_Domain_to_Codomain ( zz, fz );

   if ( Transitioning )
    {
      // In transition mode 1 we simply cross fade the colouring. We
      // don't normaly want this for transition mode 2 because it is
      // not needed and in that mode we only want one set of contour
      // lines.  But  if there were any NaNs in the function results
      // then we fall back onto this transition mode.

      if ( TransitionMode == 1 || IS_NAN(fz) )
       {
         return mix ( Codomain_Colour(fz.xy), // Previous function
                      Codomain_Colour(fz.zw), // Current function
                      TransitionFactor
                    );
       }

      // In transition mode 2 the function itself is gradualy morphed
      // from the previous function to the new function.  We can't do
      // this if either version of the function returned a Nan.

      fz.zw = mix ( fz.xy, fz.zw, TransitionFactor );

    }

   return Codomain_Colour(fz.zw);

 }

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

#endif // EVALUATE_FUNCTION

void main( void )
 {

   // If the shader has been configured to support a data panel then
   // optionally generate one.  If the only purpose if the shader is
   // to generate a data panel then it is genenerated unconditionaly
   // otherwise it is controled by the DataPanel parameter which may
   // be set in the scene file that is using this shader.

   #ifdef USE_EMU_DATA_PANEL
      #ifdef EVALUATE_FUNCTION
         vec4 colour = vec4_0;
         if ( DataPanel >= 0 ) colour = Main_Image();
         if ( DataPanel != 0 ) colour = Data_Panel(colour);
         gl_FragColor = colour;
      #else
         gl_FragColor = Data_Panel(vec4_0);
      #endif
   #else
      gl_FragColor = Main_Image();
   #endif

 }

// -----------------------------------------------------------------------------
